<?php

namespace App\Console\Commands;

use App\Models\Plan;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use LemonSqueezy\Laravel\Console\ListProductsCommand;
use LemonSqueezy\Laravel\LemonSqueezy;
use function Laravel\Prompts\spin;

class SyncPlansFromLemonSqueezy extends ListProductsCommand
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'lemonsqueezy:sync-plans
                            {--force : Force update existing plans}';

    public function handle(): int
    {
        if (! $this->validate()) {
            return static::FAILURE;
        }

        $storeResponse = spin(fn() => $this->fetchStore(), '🍋 Fetching store information...');
        $store = $storeResponse->json('data.attributes');

        return $this->handleProducts($store);
    }

    protected function handleProducts(array $store): int
    {
        $productsResponse = spin(
            fn() => LemonSqueezy::api(
                'GET',
                'products',
                [
                    'include' => 'variants',
                    'filter[store_id]' => config('lemon-squeezy.store'),
                    'page[size]' => 100,
                ],
            ),
            '🍋 Fetching products information...',
        );

        $products = collect($productsResponse->json('data'));

        $this->newLine();
        $this->displayTitle();
        $this->newLine();

        $products->each(function ($product) use ($productsResponse, $store) {
            $this->displayProduct($product);

            $variantIds = collect(Arr::get($product, 'relationships.variants.data'))->pluck('id');
            $variants = collect($productsResponse->json('included'))
                ->filter(fn($item) => $item['type'] === 'variants')
                ->filter(fn($item) => $variantIds->contains($item['id']))
                ->sortBy('sort');

            $variants->each(fn($variant) => $this->savePlan(
                $product,
                $variant,
                Arr::get($store, 'currency'),
                $variants->count() > 1,
            ));

            $this->newLine();
        });

        return static::SUCCESS;
    }

    protected function savePlan($product, $variant, $currency, $hidePending = false)
    {
        // Implement the logic to save the plan based on the variant and store information
        // This could involve creating or updating a plan in your database
        if (Arr::get($variant, 'attributes.status') === 'pending' && $hidePending) {
            return;
        }

        $variantName = Arr::get($variant, 'attributes.name');

        $price = Arr::get($variant, 'attributes.price');

        $variantId = Arr::get($variant, 'id');

        $this->info("Processing subscription price: {$variantName} (ID: {$variantId})");

        // Determine billing interval based on renewal_interval_unit
        $billingInterval = Arr::get($variant, 'attributes.interval');

        $billingInterval =  $billingInterval === 'month' ? 'monthly' : 'yearly';

        // Check if plan with this lemonsqueezy_price_id already exists
        $existingPlan = Plan::where('lemonsqueezy_price_id', $variantId)->first();

        if ($existingPlan && !$this->option('force')) {
            $this->info("Plan for price {$variantName} (ID: {$variantId}) already exists. Skipping. Use --force to update.");
            return;
        }

        // Create a unique slug based on the product name, variant name, and billing interval
        $slug = Str::slug($variantName);
        $productId = $product['id'];

        // Create or update the plan
        if ($existingPlan) {
            $existingPlan->update([
                'name' => $variantName,
                'slug' => $slug,
                'billing_interval' => $billingInterval,
                'price_cents' => $price,
                'lemonsqueezy_product_id' => $productId,
                'lemonsqueezy_price_id' => $variantId,
                'description' => $productAttributes['description'] ?? null,
                'is_active' => true,
            ]);
            $this->info("Updated existing plan: {$variantName}");
        } else {
            Plan::create([
                'name' => $variantName,
                'slug' => $slug,
                'billing_interval' => $billingInterval,
                'price_cents' => $price,
                'lemonsqueezy_product_id' => $productId,
                'lemonsqueezy_price_id' => $variantId,
                'description' => $productAttributes['description'] ?? null,
                'features' => '{
                  "Feature 1": true,
                  "Feature 2": true,
                  "Feature 3": true,
                  "Feature 4": true,
                  "Feature 5": true,
                  "Feature 6": true,
                  "Feature 7": true
                }',
                'is_active' => true,
            ]);
            $this->info("Created new plan: {$variantName}");
        }

    }

}
