<?php

namespace App\Http\Controllers;

use App\Mail\EarlyAccessConfirmation;
use App\Models\EarlyAccessSignup;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;
use Inertia\Inertia;

class EarlyAccessController extends Controller
{
    /**
     * Store a new early access signup.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'email' => 'required|email|unique:early_access_signups,email',
            ]);

            // Check if this email exists but is not verified
            $existingSignup = EarlyAccessSignup::where('email', $validated['email'])
                ->where('is_verified', false)
                ->first();

            if ($existingSignup) {
                // Regenerate token and send verification email again
                $existingSignup->update([
                    'token' => Str::uuid(),
                ]);
                
                Mail::to($existingSignup->email)->send(new EarlyAccessConfirmation($existingSignup));
                
                return Redirect::back()->with('success', 'A verification email has been sent. Please check your inbox.');
            }

            // Create new signup
            $signup = EarlyAccessSignup::create([
                'email' => $validated['email'],
                'token' => Str::uuid(),
                'is_verified' => false,
                'signed_up_at' => now(),
            ]);

            // Send verification email
            Mail::to($signup->email)->send(new EarlyAccessConfirmation($signup));

            return Redirect::back()->with('success', 'Thank you for signing up! Please check your email to confirm your address.');
        } catch (ValidationException $e) {
            return Redirect::back()->withErrors(['email' => 'This email has already been registered or is invalid.']);
        } catch (\Exception $e) {
            return Redirect::back()->withErrors(['email' => 'Something went wrong. Please try again later.']);
        }
    }
    
    /**
     * Verify an early access signup email.
     *
     * @param  string  $token
     * @return \Illuminate\Http\RedirectResponse
     */
    public function verify($token)
    {
        $signup = EarlyAccessSignup::byToken($token)->first();
        
        if (!$signup) {
            return Redirect::route('home')->with('error', 'Invalid or expired verification link.');
        }
        
        // Check if already verified
        if ($signup->is_verified) {
            return Redirect::route('home')->with('success', 'Your email has already been verified. Thank you!');
        }
        
        // Verify the signup
        $signup->update([
            'is_verified' => true,
            'verified_at' => now(),
            'token' => null, // Clear the token for security
        ]);
        
        return Redirect::route('home')->with('success', 'Your email has been verified. Thank you for confirming your early access request!');
    }
}
