<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Plan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'billing_interval',
        'price_cents',
        'lemonsqueezy_product_id',
        'lemonsqueezy_price_id',
        'is_active',
        'description',
        'features',
    ];

    protected $casts = [
        'features' => 'array',
        'is_active' => 'boolean',
        'price_cents' => 'integer',
    ];
    
    /**
     * Get the price in dollars
     *
     * @return float
     */
    public function getPriceDollarsAttribute(): float
    {
        return $this->price_cents / 100;
    }
    
    /**
     * Set the price in dollars
     *
     * @param float|string $value
     * @return void
     */
    public function setPriceDollarsAttribute($value): void
    {
        $this->attributes['price_cents'] = (int)($value * 100);
    }
    
    /**
     * Prepare features for Filament Repeater
     *
     * @param mixed $value
     * @return array
     */
    public function getFeaturesAttribute($value)
    {
        $decodedFeatures = json_decode($value, true) ?: [];
        
        if (!is_array($decodedFeatures)) {
            return [];
        }
        
        // Transform features into array of objects for Repeater
        $formattedFeatures = [];
        
        // Handle object format (key => boolean)
        if (!isset($decodedFeatures[0])) { // Not a sequential array
            foreach ($decodedFeatures as $feature => $enabled) {
                $formattedFeatures[] = [
                    'feature' => $feature,
                    'enabled' => (bool)$enabled
                ];
            }
        } 
        // Handle array format (for backward compatibility)
        else {
            foreach ($decodedFeatures as $feature) {
                if (is_string($feature)) {
                    $formattedFeatures[] = ['feature' => $feature, 'enabled' => true];
                }
            }
        }
        
        return $formattedFeatures;
    }
    
    /**
     * Save features from Filament form
     *
     * @param mixed $value
     * @return void
     */
    public function setFeaturesAttribute($value)
    {
        $features = [];
        
        // Handle array of objects from Filament Repeater
        if (is_array($value)) {
            // Convert to object format with feature name as key and enabled as value
            foreach ($value as $item) {
                if (is_array($item) && isset($item['feature'])) {
                    $featureName = $item['feature'];
                    $enabled = isset($item['enabled']) ? (bool)$item['enabled'] : true;
                    $features[$featureName] = $enabled;
                }
            }
            $this->attributes['features'] = json_encode($features);
        } elseif (is_string($value) && !empty($value)) {
            // If it's already a JSON string, store it directly
            $this->attributes['features'] = $value;
        } else {
            // Default to empty object
            $this->attributes['features'] = json_encode((object)[]);
        }
    }

    /**
     * Get the formatted price with currency symbol
     * 
     * @return string
     */
    public function formattedPrice(): string
    {
        return '$' . number_format($this->price_cents / 100, 2);
    }

    /**
     * Get the billing interval with proper formatting
     * 
     * @return string
     */
    public function formattedInterval(): string
    {
        return $this->billing_interval === 'monthly' ? 'month' : 'year';
    }
}
