import { useState, useRef, useEffect } from 'react';
import { useForm, usePage } from '@inertiajs/react';
import { motion, AnimatePresence } from 'framer-motion';
import { type SharedData } from '@/types';

type FlashMessageType = {
    success?: string;
    error?: string;
    message?: string;
    errors?: {
        email?: string;
    };
};

interface EmailSignupSectionProps {
    emailSectionRef?: React.RefObject<HTMLDivElement>;
}

export default function EmailSignupSection({ emailSectionRef }: EmailSignupSectionProps) {
    const pageProps = usePage<SharedData & { flash?: FlashMessageType }>().props;
    const flash = pageProps.flash || {} as FlashMessageType;
    const [emailSubmitted, setEmailSubmitted] = useState(false);
    const localEmailSectionRef = useRef<HTMLDivElement | null>(null);
    
    // Use provided ref or local ref
    const sectionRef = emailSectionRef || localEmailSectionRef;

    const { data, setData, post, processing, errors, reset } = useForm({
        email: '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post(route('early-access.store'), {
            preserveScroll: true,
            onSuccess: () => {
                setEmailSubmitted(true);
                reset('email');
            },
        });
    };

    // Ensure scroll position stays at the email form after submission
    useEffect(() => {
        if (emailSubmitted && sectionRef.current) {
            sectionRef.current.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    }, [emailSubmitted, sectionRef]);

    return (
        <section ref={sectionRef} className="py-16 md:py-20 bg-gradient-to-b from-indigo-50 to-white dark:from-indigo-950/20 dark:to-gray-900">
            <div className="container px-4 md:px-6 mx-auto max-w-7xl">
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5 }}
                    className="flex flex-col items-center text-center space-y-4 md:space-y-6 max-w-3xl mx-auto">
                    <div className="inline-block rounded-full bg-indigo-100 px-3 py-1 text-sm font-medium text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300">
                        Limited Time Offer
                    </div>
                    <h2 className="text-3xl font-bold tracking-tight sm:text-4xl md:text-5xl font-jakarta">Get Notified When We Launch</h2>
                    <p className="text-gray-500 md:text-xl/relaxed max-w-[600px] dark:text-gray-400">
                        Join our waitlist to be the first to know when we launch and receive special early-bird pricing.
                    </p>

                    <div className="w-full max-w-md mt-6">
                        <AnimatePresence mode="wait">
                            {emailSubmitted ? (
                                <motion.div
                                    key="success"
                                    initial={{ opacity: 0, y: 10 }}
                                    animate={{ opacity: 1, y: 0 }}
                                    exit={{ opacity: 0, y: -10 }}
                                    className="bg-green-50 border border-green-200 rounded-lg p-4 text-green-800 dark:bg-green-900/20 dark:border-green-800 dark:text-green-300"
                                >
                                    <div className="flex items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2" viewBox="0 0 20 20" fill="currentColor">
                                            <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                                        </svg>
                                        <p>{(flash && flash.success) ? flash.success : 'Thank you for signing up! Please check your inbox to verify your email address.'}</p>
                                    </div>
                                </motion.div>
                            ) : (
                                <motion.form
                                    key="form"
                                    initial={{ opacity: 0, y: 10 }}
                                    animate={{ opacity: 1, y: 0 }}
                                    exit={{ opacity: 0, y: -10 }}
                                    onSubmit={handleSubmit}
                                    className="space-y-3"
                                >
                                    <div className="flex rounded-lg shadow-sm">
                                        <input
                                            type="email"
                                            name="email"
                                            placeholder="Enter your email"
                                            value={data.email}
                                            onChange={(e) => setData('email', e.target.value)}
                                            className="flex-1 rounded-l-lg border border-gray-300 bg-white px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-indigo-500 dark:border-gray-700 dark:bg-gray-950"
                                            required
                                        />
                                        <button
                                            type="submit"
                                            disabled={processing}
                                            className="inline-flex items-center justify-center rounded-r-lg border border-indigo-600 bg-indigo-600 px-5 py-2 text-sm font-medium text-white hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 disabled:opacity-70 dark:border-indigo-500 dark:bg-indigo-500 dark:hover:bg-indigo-600"
                                        >
                                            {processing ? (
                                                <svg className="h-4 w-4 animate-spin" viewBox="0 0 24 24">
                                                    <circle
                                                        className="opacity-25"
                                                        cx="12"
                                                        cy="12"
                                                        r="10"
                                                        stroke="currentColor"
                                                        strokeWidth="4"
                                                        fill="none"
                                                    />
                                                    <path
                                                        className="opacity-75"
                                                        fill="currentColor"
                                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                    />
                                                </svg>
                                            ) : (
                                                'Notify Me'
                                            )}
                                        </button>
                                    </div>
                                    {errors.email && (
                                        <div className="text-sm text-red-600 dark:text-red-400">{errors.email}</div>
                                    )}
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        We'll send you a verification link to confirm your email address.
                                    </p>
                                </motion.form>
                            )}
                        </AnimatePresence>
                    </div>
                </motion.div>
            </div>
        </section>
    );
}
