import React from 'react';
import { motion } from 'framer-motion';
import { type Plan } from '@/hooks/usePlans';
import FeatureItem from './FeatureItem';

type PlanCardProps = {
    plan: Plan;
    billingInterval: 'monthly' | 'yearly';
    auth: { user: any /* eslint-disable-line @typescript-eslint/no-explicit-any */ } | null;
};

const PlanCard: React.FC<PlanCardProps> = ({ plan, billingInterval, auth }) => {
    const features = plan.features;

    return (
        <motion.div
            key={plan.id}
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5 }}
            className="relative rounded-2xl border border-gray-200 bg-white p-6 shadow-sm dark:border-gray-800 dark:bg-gray-900 flex flex-col h-full"
        >
            {plan.name.toLowerCase().includes('pro') && (
                <div className="absolute top-0 right-0 -translate-y-1/2 translate-x-0 rounded-full bg-green-100 px-3 py-1 text-xs font-medium text-green-800 dark:bg-green-900/30 dark:text-green-300">
                    Best Value
                </div>
            )}
            <div className="mb-5">
                <h3 className="text-lg font-semibold">{plan.name}</h3>
                <div className="mt-2 flex items-baseline">
                    <span className="text-3xl font-bold">${(plan.price_cents / 100).toFixed(2)}</span>
                    <span className="text-gray-500 dark:text-gray-400 ml-1">/{plan.billing_interval === 'monthly' ? 'month' : 'year'}</span>
                </div>
                <p className="mt-3 text-sm text-gray-500 dark:text-gray-400">{plan.description}</p>
                {billingInterval === 'yearly' && (
                    <p className="mt-1 text-xs text-green-600 dark:text-green-400">
                        Includes 4 months free
                    </p>
                )}
            </div>

            <div className="mt-2 space-y-3 flex-grow">
                <h4 className="text-sm font-medium">What's included:</h4>
                <ul className="space-y-3">
                    {Array.isArray(features) && features.length > 0 ? (
                        features.map((feature, index) => (
                            <FeatureItem
                                key={index}
                                feature={feature.feature}
                                enabled={feature.enabled}
                            />
                        ))
                    ) : (
                        <li className="text-sm text-gray-500">No features available</li>
                    )}
                </ul>
            </div>

            <div className="mt-6">
                <a
                    href={auth && auth.user
                        ? route('subscription.checkout', plan.id)
                        : route('register', { plan_id: plan.id })}
                    className="inline-flex w-full items-center justify-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
                >
                    Get Started
                </a>
            </div>
        </motion.div>
    );
};

export default PlanCard;
