import { router, usePage } from '@inertiajs/react';
import { Head } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import { useState } from 'react';
import { motion } from 'framer-motion';
import BillingToggle from '@/components/subscription/BillingToggle';
import PlanCard from '@/components/subscription/PlanCard';

type Plan = {
  id: number;
  name: string;
  slug: string;
  billing_interval: string;
  price_cents: number;
  description?: string;
  features: string[];
  is_active: boolean;
  formattedPrice: string;
  formattedInterval: string;
  lemonsqueezy_price_id: string;
  lemonsqueezy_product_id: string;
};

type Subscription = {
  id: number;
  billable_type: string;
  billable_id: number;
  type: string;
  lemon_squeezy_id: string;
  status: string;
  product_id: string;
  variant_id: string;
  card_brand?: string;
  card_last_four?: string;
  pause_mode: string | null;
  pause_resumes_at: string | null;
  trial_ends_at: string | null;
  renews_at?: string;
  ends_at?: string;
  created_at: string;
  updated_at: string;
  active: boolean;
  cancelled: boolean;
  ended: () => boolean;
};

interface FlashMessage {
  success?: string;
  error?: string;
}

interface SubscriptionPageProps {
  subscriptions: Subscription[];
  activeSubscription: Subscription | null;
  plans: Plan[];
  flash: FlashMessage;
  [key: string]: unknown;
}

export default function SubscriptionManage() {
  const { activeSubscription, plans, flash } = usePage<SubscriptionPageProps>().props;
  const [confirmingCancel, setConfirmingCancel] = useState<boolean>(false);
  const [currentSubscriptionId, setCurrentSubscriptionId] = useState<number | null>(null);
  const [billingInterval, setBillingInterval] = useState<'monthly' | 'yearly'>('monthly');

  const handleCancelSubscription = (subscriptionId: number) => {
    setCurrentSubscriptionId(subscriptionId);
    setConfirmingCancel(true);
  };

  const confirmCancel = () => {
    router.post('/subscription/cancel', {
      subscription_id: currentSubscriptionId
    });
    setConfirmingCancel(false);
  };

  const handleResumeSubscription = (subscriptionId: number) => {
    router.post('/subscription/resume', {
      subscription_id: subscriptionId
    });
  };

  const handleChangePlan = (planId: number) => {
    if (activeSubscription) {
      router.post('/subscription/change-plan', {
        subscription_id: activeSubscription.id,
        plan_id: planId
      });
    }
  };

  const breadcrumbs = [
    {
      title: 'Subscription',
      href: '/subscription',
    },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Subscription Management" />
      <div className="container mx-auto p-8">
        {flash.success && (
          <Alert className="mb-6 bg-green-50">
            <AlertTitle>Success</AlertTitle>
            <AlertDescription>{flash.success}</AlertDescription>
          </Alert>
        )}

        {flash.error && (
          <Alert className="mb-6 bg-red-50">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{flash.error}</AlertDescription>
          </Alert>
        )}

        {/* Current Subscription Summary */}
        {activeSubscription && (
          <Card className="mb-8 border-indigo-200 bg-indigo-50/50 dark:bg-indigo-900/10 dark:border-indigo-900/20">
            <CardHeader>
              <CardTitle>Your Active Subscription</CardTitle>
              <CardDescription>Manage your current subscription details</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {/* Plan Name */}
                <div className="flex justify-between border-b border-indigo-100 dark:border-indigo-900/30 pb-3">
                  <span className="font-medium">Plan:</span>
                  <span className="font-semibold text-indigo-600 dark:text-indigo-400">
                    {plans.find(plan => plan.lemonsqueezy_product_id === activeSubscription.product_id)?.name || 'Unknown Plan'}
                  </span>
                </div>
                <div className="flex justify-between border-b border-indigo-100 dark:border-indigo-900/30 pb-3">
                  <span className="font-medium">Status:</span>
                  <span className="px-2 py-1 rounded-full bg-green-100 text-green-800 text-xs dark:bg-green-900/30 dark:text-green-300">
                    {activeSubscription.cancelled ? "Cancelled" : "Active"}
                  </span>
                </div>
                {activeSubscription.renews_at && (
                  <div className="flex justify-between border-b border-indigo-100 dark:border-indigo-900/30 pb-3">
                    <span className="font-medium">Renews on:</span>
                    <span>{new Date(activeSubscription.renews_at).toLocaleDateString()}</span>
                  </div>
                )}
                {activeSubscription.ends_at && (
                  <div className="flex justify-between border-b border-indigo-100 dark:border-indigo-900/30 pb-3">
                    <span className="font-medium">Ends on:</span>
                    <span>{new Date(activeSubscription.ends_at).toLocaleDateString()}</span>
                  </div>
                )}
                <div className="flex justify-between">
                  <span className="font-medium">Created on:</span>
                  <span>{new Date(activeSubscription.created_at).toLocaleDateString()}</span>
                </div>
              </div>
            </CardContent>
            <CardFooter className="flex justify-end space-x-2">
              {activeSubscription.cancelled && activeSubscription.ends_at && new Date(activeSubscription.ends_at) > new Date() && (
                <Button
                  onClick={() => handleResumeSubscription(activeSubscription.id)}
                  variant="outline"
                >
                  Resume Subscription
                </Button>
              )}

              {!activeSubscription.cancelled && (
                <Button
                  onClick={() => handleCancelSubscription(activeSubscription.id)}
                  variant="destructive"
                >
                  Cancel Subscription
                </Button>
              )}
            </CardFooter>
          </Card>
        )}

        {/* Pricing Section */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
          className="text-center mb-8"
        >
          <h2 className="text-3xl font-bold tracking-tight sm:text-4xl font-jakarta mb-4">
            {activeSubscription ? "Change Your Plan" : "Choose Your Plan"}
          </h2>
          <p className="text-gray-500 md:text-xl/relaxed lg:text-base/relaxed xl:text-xl/relaxed dark:text-gray-400 max-w-3xl mx-auto">
            {activeSubscription
              ? "Upgrade or downgrade your subscription at any time. Changes will take effect at the end of your current billing period."
              : "Select the plan that works best for you. All plans include access to all features."}
          </p>
        </motion.div>

        {/* Billing Interval Toggle */}
        <BillingToggle
          billingInterval={billingInterval}
          setBillingInterval={setBillingInterval}
          yearlySavingsPercentage={33.33} // Approximately 4 months free
        />

        {/* Filter plans by billing interval */}
        {(() => {
          const filteredPlans = plans.filter(plan => plan.billing_interval === billingInterval);

          if (filteredPlans.length === 0) {
            return (
              <div className="text-center py-8">
                <p className="text-gray-500 dark:text-gray-400">
                  No {billingInterval} pricing plans are currently available.
                </p>
              </div>
            );
          }

          return (
            <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:grid-cols-3 lg:gap-8 max-w-7xl mx-auto">
              {filteredPlans.map((plan) => (
                <PlanCard
                  key={plan.id}
                  plan={plan}
                  billingInterval={billingInterval}
                  activeSubscription={activeSubscription}
                  onChangePlan={handleChangePlan}
                />
              ))}
            </div>
          );
        })()}

        {/* Confirmation Modal */}
        {confirmingCancel && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white dark:bg-gray-900 p-6 rounded-lg max-w-md w-full shadow-xl">
              <h3 className="text-xl font-bold mb-4">Confirm Cancellation</h3>
              <p className="mb-6 text-gray-600 dark:text-gray-300">
                Are you sure you want to cancel your subscription? You'll continue to have access until the end of your current billing period.
              </p>
              <div className="flex justify-end space-x-3">
                <Button variant="outline" onClick={() => setConfirmingCancel(false)}>
                  Nevermind
                </Button>
                <Button variant="destructive" onClick={confirmCancel}>
                  Yes, Cancel Subscription
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </AppLayout>
  );
}
