<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Webklex\PHPIMAP\ClientManager;
use Webklex\IMAP\Facades\Client;
use Illuminate\Support\Carbon;
use App\Models\{Tickets,Responses,User,Departments,MailAccount};

class FetchEmails extends Command
{
    protected $signature = 'emails:fetch {--account= : Comma-separated list of account IDs} {--all : Fetch from all active accounts}';
    protected $description = 'Fetch emails from one or more IMAP accounts (from database)';

    public function handle()
    {
        $mailAccounts = [];

        if ($this->option('all')) {
            // Fetch from all active accounts
            $mailAccounts = MailAccount::active()->get();
            $this->info("Fetching emails from all active accounts...");
        } elseif ($this->option('account')) {
            // Fetch from specific account IDs
            $accountIds = explode(',', $this->option('account'));
            $mailAccounts = MailAccount::whereIn('id', $accountIds)->active()->get();
        } else {
            // Fetch from the default account
            $mailAccounts = MailAccount::default()->active()->get();

            if ($mailAccounts->isEmpty()) {
                $this->info("No default account found. Fetching from all active accounts...");
                $mailAccounts = MailAccount::active()->get();
            }
        }

        if ($mailAccounts->isEmpty()) {
            $this->warn('No active mail accounts found. Please create a mail account in Settings > Mail Settings.');
            return Command::FAILURE;
        }

        foreach ($mailAccounts as $mailAccount) {
            $this->fetchFromAccount($mailAccount);
        }

        return Command::SUCCESS;
    }

    protected function stripQuotedReply($body)
    {
        if (empty($body)) {
            return $body;
        }

        $lines = explode("\n", $body);
        $cleanedLines = [];
        $inQuote = false;

        foreach ($lines as $line) {
            $trimmedLine = ltrim($line);

            // Check if this line starts a quoted section (>, |, etc.)
            if (preg_match('/^[>|]/', $trimmedLine)) {
                $inQuote = true;
                continue;
            }

            // Check for common reply headers like "On ... wrote:", "From:", etc.
            if (preg_match('/^(On\s+.*wrote:|From:|Sent:|Date:|To:|Subject:|--)/i', $trimmedLine)) {
                $inQuote = true;
                continue;
            }

            // If we hit a quoted section, stop including content
            if ($inQuote) {
                continue;
            }

            $cleanedLines[] = $line;
        }

        // Trim trailing empty lines
        while (!empty($cleanedLines) && trim(end($cleanedLines)) === '') {
            array_pop($cleanedLines);
        }

        return implode("\n", $cleanedLines);
    }

    protected function fetchFromAccount(MailAccount $mailAccount)
    {
        $this->info("Connecting to account: {$mailAccount->name} ({$mailAccount->email})");

        try {
            // Build IMAP configuration from database
            $imapConfig = [
                'accounts' => [
                    'default' => $mailAccount->getImapConfigArray()
                ],
                'options' => config('imap.options'),
                'date_format' => config('imap.date_format'),
                'security' => config('imap.security'),
                'decoding' => config('imap.decoding'),
                'flags' => config('imap.flags'),
                'events' => config('imap.events'),
                'masks' => config('imap.masks'),
            ];

            $cm = new ClientManager($imapConfig);
            $client = $cm->account('default');
            $client->connect();
        } catch (\Exception $e) {
            $this->error("Failed to connect to account {$mailAccount->name}: ".$e->getMessage());
            return;
        }

        $folder = $client->getFolder('INBOX');
        $messages = $folder->messages()->unseen()->limit(10)->get();

        foreach ($messages as $message) {

            // Use envelope to get reliable recipients
            $envelope = $message->getEnvelope();
            $toAddresses = [];

            if (!empty($envelope->to)) {
                foreach ($envelope->to as $addr) {
                    $toAddresses[] = $addr->mailbox . '@' . $addr->host;
                }
            }

            // Fallback to header parsing if envelope empty
            if (empty($toAddresses)) {
                $toAddresses = collect($message->getTo())->pluck('mail')->toArray();
            }

            $to = implode(',', $toAddresses);
            $from = $envelope->fromaddress ?? ($message->getFrom()[0]->mail ?? null);

            // Get sender's name and email from email message - try multiple sources
            $fromName = null;
            $fromEmail = $message->getFrom()[0]->mail ?? null;

            // Try to get name from various email header sources
            if (!empty($message->getFrom()[0]->personal)) {
                $fromName = $message->getFrom()[0]->personal;
            } elseif (!empty($message->getFrom()[0]->name)) {
                $fromName = $message->getFrom()[0]->name;
            } elseif (!empty($envelope->from) && !empty($envelope->from[0]->personal)) {
                $fromName = $envelope->from[0]->personal;
            } elseif (!empty($envelope->sender) && !empty($envelope->sender[0]->personal)) {
                $fromName = $envelope->sender[0]->personal;
            }

            // If we still don't have a name, try parsing it from the full from address
            if (empty($fromName) && !empty($from) && strpos($from, '<') !== false) {
                // Extract name from "Name <email@example.com>" format
                if (preg_match('/^(.+?)\s*</', $from, $matches)) {
                    $fromName = trim($matches[1], '"\'');
                }
            }

            // Final fallback: extract username from email address (e.g., "mike" from "mike@thedances.co.uk")
            if (empty($fromName) && !empty($fromEmail)) {
                $emailParts = explode('@', $fromEmail);
                $fromName = ucfirst($emailParts[0]); // Capitalize first letter
            }

            $this->line("From: {$from}");
            $this->line("From Name: " . ($fromName ?? 'Not found'));
            $this->line("To:   {$to}");
            $this->line("Subject: ".$message->getSubject());

            $check_user = User::where('email', '=', $message->getFrom()[0]->mail)->first();

            $subject = $message->getSubject();

            // Strip Re:, Fwd:, etc. prefixes
            do {
                $new = preg_replace('/^(Re:|Fwd:|FW:|Aw:|Tr:)\s*/i', '', $subject);
                $changed = ($new !== $subject);
                $subject = $new;
            } while ($changed);

            // Remove trailing fullstop/period from subject
            $subject = rtrim($subject, '.');

            // Initialize employee_response - default to '0' (not an employee)
            $employee_response = '0';

            if($check_user)
            {
                // For registered users: match by subject and user_id
                $check_ticket = Tickets::where('subject', '=', $subject)->where('user_id', '=', $check_user->id)->first();

                if($check_user->hasAnyRole('admin', 'employee'))
                {
                    $employee_response = '1';
                } else {
                    $employee_response = '0';
                }
            } else {
                // For guest users: match by subject and public_email
                $check_ticket = Tickets::where('subject', '=', $subject)
                    ->where('user_id', '=', '0')
                    ->where('public_email', '=', $message->getFrom()[0]->mail)
                    ->first();
            }

            if ($check_ticket) {
                // Strip quoted reply content from the message body
                $cleanedBody = $this->stripQuotedReply($message->getTextBody());

                $response = Responses::create([
                    'user_id' => $check_ticket->user_id ?? '0',
                    'content' => $cleanedBody,
                    'ticket_number' => $check_ticket->id,
                    'is_note' => 0,
                    'organize' => 'email',
                    'employee_response' => $employee_response,
                    'ip_address' => null,
                    'created_at' => Carbon::now(),
                    'updated_at' => null,
                ]);
                
                if ($employee_response == '1') {
                    Tickets::where('id', '=', $check_ticket->id)
                        ->update([
                            'updated_at' => Carbon::now(),
                        ]);

                    $first_check = Tickets::where('id', '=', $check_ticket->id)->first();
                    if($first_check->first_employee_reply == '')
                    {
                        if($check_ticket->status == 'in progress') {
                            Tickets::where('id', '=', $check_ticket->id)
                                ->update([
                                    'first_employee_reply' => Carbon::now(),
                                ]);
                        } else {
                            Tickets::where('id', '=', $check_ticket->id)
                                ->update([
                                    'first_employee_reply' => Carbon::now(),
                                    'status' => 'awaiting reply',
                                ]);
                        }
                    }

                } else {
                    if($check_ticket->status == 'in progress') {
                        Tickets::where('id', '=', $check_ticket->id)
                            ->update([
                                'date_closed' => null,
                                'updated_by_client_at' => Carbon::now(),
                            ]);
                    } else {
                        Tickets::where('id', '=', $check_ticket->id)
                            ->update([
                                'date_closed' => null,
                                'updated_by_client_at' => Carbon::now(),
                                'status' => 'open',
                            ]);
                    }

                }
            } else {
                // Strip quoted reply content from the message body
                $cleanedBody = $this->stripQuotedReply($message->getTextBody());

                $token = bin2hex(random_bytes(16));
                $public_hash = hash_hmac('sha256', $token, config('app.key'));

                $department = Departments::select('id', 'is_public', 'is_disabled', 'soft_deleted')->where('department_email', $to)->first();
                if(!$department) {
                    $department_id = '1';
                    $department_is_public = 1; // Default department is public
                } else {
                    $department_id = $department['id'];
                    $department_is_public = $department['is_public'];

                    // Reject emails sent to disabled departments
                    if ($department['is_disabled'] == 1) {
                        $this->warn("Skipping email: Department is disabled.");
                        $message->setFlag('Seen');
                        continue;
                    }

                    // Reject emails sent to soft deleted departments
                    if ($department['soft_deleted'] == 1) {
                        $this->warn("Skipping email: Department is being deleted.");
                        $message->setFlag('Seen');
                        continue;
                    }
                }

                // Check if user exists
                $check_user = User::where('email', '=', $message->getFrom()[0]->mail)->first();

                // If department is customers only (is_public = 2), reject guests (only registered users allowed)
                if ($department_is_public == 2 && !$check_user) {
                    $this->warn("Skipping email: Department is customers only and sender is not a registered user.");
                    $message->setFlag('Seen');
                    continue;
                }
                if($check_user)
                {
                    // Create a Ticket for an existing user.
                    $account_manager = User::select('account_manager')->where('id', $check_user->id)->first();
                    $ticket = Tickets::create([
                        'user_id' => $check_user->id ?? '0',
                        'subject' => $message->getSubject() ?? 'No Subject',
                        'message' => $cleanedBody,
                        'status' => 'open',
                        'priority' => 'low',
                        'cc' => '',
                        'assigned' => $account_manager["account_manager"] ?? '0',
                        'department_id' => $department_id,
                        'public_hash' => $public_hash,
                        'public_name' => $check_user->name ?? null,
                        'public_email' => $check_user->email ?? null,
                        'rating' => '0',
                        'ip_address' => app('request')->ip(),
                        'organize' => 'email',
                        'date_closed' => null,
                        'updated_by_client_at' => date("Y-m-d H:i:s"),
                        'created_at' => date("Y-m-d H:i:s"),
                        'updated_at' => null,
                        'first_employee_reply' => null,
                    ]);
                } else {
                    // Create a Ticket as a Guest.
                    $ticket = Tickets::create([
                        'user_id' => '0',
                        'subject' => $message->getSubject() ?? 'No Subject',
                        'message' => $cleanedBody,
                        'status' => 'open',
                        'priority' => 'low',
                        'cc' => '',
                        'assigned' => '0',
                        'department_id' => $department_id,
                        'public_hash' => $public_hash,
                        'public_name' => $fromName ?? $from ?? 'Unknown',
                        'public_email' => $from ?? null,
                        'rating' => '0',
                        'ip_address' => app('request')->ip(),
                        'organize' => 'email',
                        'date_closed' => null,
                        'updated_by_client_at' => date("Y-m-d H:i:s"),
                        'created_at' => date("Y-m-d H:i:s"),
                        'updated_at' => null,
                        'first_employee_reply' => null,
                    ]);
                }

            
                /* Email::create([
                    'from'    => $message->getFrom()[0]->mail,
                    'to'      => $toAddresses,
                    'subject' => $message->getSubject(),
                    'body'    => $message->getTextBody(),
                    'account' => $account, // optional: track source
                ]);*/
            }

            // Mark as seen
            $message->setFlag('Seen');
        }

        $client->disconnect();

    }
}