<?php

namespace App\Console\Commands;

use App\Models\Departments;
use App\Models\User;
use App\Services\Installation\InstallationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class TicagaInstall extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ticaga:install
                            {--force : Force installation even if already installed}
                            {--skip-migrations : Skip running database migrations}
                            {--skip-dependencies : Skip installing dependencies}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install Ticaga with interactive prompts';

    protected InstallationService $installation;

    public function __construct(InstallationService $installation)
    {
        parent::__construct();
        $this->installation = $installation;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('');
        $this->info('╔════════════════════════════════════════════╗');
        $this->info('║      Ticaga Installation Wizard            ║');
        $this->info('║      Version ' . config('ticaga.version', '1.0.0') . '                             ║');
        $this->info('╚════════════════════════════════════════════╝');
        $this->info('');

        // Check if already installed
        if ($this->installation->isInstalled() && ! $this->option('force')) {
            $this->error('✗ Ticaga is already installed!');
            $this->info('  Use --force to reinstall (this will NOT delete existing data)');

            return self::FAILURE;
        }

        // Step 1: System Requirements
        $this->newLine();
        $this->info('Step 1/5: Checking System Requirements...');
        if (! $this->checkRequirements()) {
            return self::FAILURE;
        }

        // Step 2: Dependencies
        $this->newLine();
        $this->info('Step 2/5: Checking Dependencies...');
        if (! $this->option('skip-dependencies')) {
            if (! $this->checkDependencies()) {
                return self::FAILURE;
            }
        } else {
            $this->warn('  ⚠ Skipping dependency check (--skip-dependencies)');
        }

        // Step 3: Database Configuration
        $this->newLine();
        $this->info('Step 3/5: Database Configuration...');
        if (! $this->configureDatabaseInteractive()) {
            return self::FAILURE;
        }

        // Step 4: Run Migrations
        $this->newLine();
        $this->info('Step 4/5: Setting up Database...');
        if (! $this->option('skip-migrations')) {
            if (! $this->runMigrations()) {
                return self::FAILURE;
            }
        } else {
            $this->warn('  ⚠ Skipping migrations (--skip-migrations)');
        }

        // Step 5: Create Administrator
        $this->newLine();
        $this->info('Step 5/5: Creating Administrator Account...');
        if (! $this->createAdministrator()) {
            return self::FAILURE;
        }

        // Mark installation complete
        $this->installation->markInstallationComplete();

        $this->newLine();
        $this->info('╔════════════════════════════════════════════╗');
        $this->info('║   ✓ Installation Complete!                ║');
        $this->info('╚════════════════════════════════════════════╝');
        $this->newLine();
        $this->info('You can now log in at: ' . config('app.url'));
        $this->info('Run "php artisan ticaga:health" to verify installation');
        $this->newLine();

        return self::SUCCESS;
    }

    protected function checkRequirements(): bool
    {
        $requirements = $this->installation->systemRequirements();
        $passed = true;

        foreach ($requirements as $requirement) {
            if ($requirement['passed']) {
                $this->info('  ✓ ' . $requirement['label']);
            } elseif ($requirement['blocking']) {
                $this->error('  ✗ ' . $requirement['label'] . ' - ' . $requirement['details']);
                $passed = false;
            } else {
                $this->warn('  ⚠ ' . $requirement['label'] . ' - ' . $requirement['details']);
            }
        }

        if (! $passed) {
            $this->error('Please fix blocking requirements before continuing.');
        }

        return $passed;
    }

    protected function checkDependencies(): bool
    {
        if (! $this->installation->dependenciesSatisfied()) {
            $this->warn('  ⚠ Dependencies are not installed');

            if ($this->confirm('Would you like to install dependencies now?', true)) {
                $this->info('  Installing dependencies... (this may take several minutes)');

                try {
                    $this->installation->ensureDependenciesInstalled();
                    $this->info('  ✓ Dependencies installed successfully');

                    return true;
                } catch (\Throwable $e) {
                    $this->error('  ✗ Failed to install dependencies: ' . $e->getMessage());

                    return false;
                }
            } else {
                $this->error('  ✗ Dependencies are required for Ticaga to function');

                return false;
            }
        }

        $this->info('  ✓ All dependencies are installed');

        return true;
    }

    protected function configureDatabaseInteractive(): bool
    {
        // Try existing config first
        if ($this->installation->databaseIsReachable()) {
            $this->info('  ✓ Database connection is already configured');

            if (! $this->confirm('Would you like to reconfigure the database?', false)) {
                return true;
            }
        }

        $dbHost = $this->ask('Database Host', config('database.connections.mysql.host', '127.0.0.1'));
        $dbPort = $this->ask('Database Port', config('database.connections.mysql.port', '3306'));
        $dbName = $this->ask('Database Name', config('database.connections.mysql.database', 'ticaga'));
        $dbUser = $this->ask('Database Username', config('database.connections.mysql.username', 'root'));
        $dbPass = $this->secret('Database Password');

        // Test connection
        $configuration = [
            'host' => $dbHost,
            'port' => $dbPort,
            'database' => $dbName,
            'username' => $dbUser,
            'password' => $dbPass,
        ];

        try {
            $this->installation->testDatabaseConnection($configuration);
            $this->info('  ✓ Database connection successful');

            // Update .env file
            $this->installation->updateEnvironmentFile([
                'DB_HOST' => $dbHost,
                'DB_PORT' => $dbPort,
                'DB_DATABASE' => $dbName,
                'DB_USERNAME' => $dbUser,
                'DB_PASSWORD' => $dbPass,
            ]);

            // Update runtime config
            config([
                'database.connections.mysql.host' => $dbHost,
                'database.connections.mysql.port' => $dbPort,
                'database.connections.mysql.database' => $dbName,
                'database.connections.mysql.username' => $dbUser,
                'database.connections.mysql.password' => $dbPass,
            ]);

            DB::purge('mysql');
            DB::reconnect('mysql');

            return true;
        } catch (\Throwable $e) {
            $this->error('  ✗ Database connection failed: ' . $e->getMessage());

            if ($this->confirm('Would you like to try again?', true)) {
                return $this->configureDatabaseInteractive();
            }

            return false;
        }
    }

    protected function runMigrations(): bool
    {
        try {
            $this->info('  Running migrations...');
            Artisan::call('migrate', ['--force' => true], $this->output);

            $this->info('  Running database seeders...');
            Artisan::call('db:seed', ['--force' => true], $this->output);

            $this->info('  ✓ Database setup complete');

            return true;
        } catch (\Throwable $e) {
            $this->error('  ✗ Migration failed: ' . $e->getMessage());
            $this->error('  Please ensure the database user has CREATE, ALTER, DROP, INSERT privileges');

            return false;
        }
    }

    protected function createAdministrator(): bool
    {
        if ($this->installation->hasUsers()) {
            $this->warn('  ⚠ Users already exist in the database');

            if (! $this->confirm('Would you like to create an additional administrator?', false)) {
                $this->info('  ✓ Skipping administrator creation');

                return true;
            }
        }

        $name = $this->ask('Administrator Name', 'Admin');
        $email = $this->ask('Administrator Email');

        // Validate email
        if (! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->error('  ✗ Invalid email address');

            return $this->createAdministrator();
        }

        // Check if email already exists
        if (User::where('email', $email)->exists()) {
            $this->error('  ✗ A user with this email already exists');

            return $this->createAdministrator();
        }

        $password = $this->secret('Administrator Password (min 15 characters)');
        $passwordConfirm = $this->secret('Confirm Password');

        if ($password !== $passwordConfirm) {
            $this->error('  ✗ Passwords do not match');

            return $this->createAdministrator();
        }

        if (strlen($password) < 15) {
            $this->error('  ✗ Password must be at least 15 characters');

            return $this->createAdministrator();
        }

        try {
            DB::beginTransaction();

            $user = User::create([
                'name' => $name,
                'email' => $email,
                'company' => '',
                'password' => Hash::make($password),
                'billing_id' => '0',
                'billing_system' => '',
                'account_manager' => '0',
                'biography' => '',
                'force_password_reset' => false,
            ]);

            $user->forceFill([
                'phone_number' => '',
                'email_verified_at' => now(),
            ])->save();

            $user->assignRole('superadmin');

            $token = $user->createToken('Primary API Token')->plainTextToken;
            $user->update(['api_token' => $token]);

            // Create default department if none exists
            if (! Departments::query()->exists()) {
                $appUrl = parse_url(config('app.url'), PHP_URL_HOST) ?? 'localhost';
                $defaultEmail = 'noreply@' . $appUrl;

                Departments::create([
                    'department_name' => 'Support',
                    'department_description' => 'Default support department',
                    'slug' => Str::slug('support'),
                    'allows_high_priority' => 1,
                    'cc_enabled' => 1,
                    'is_public' => 1,
                    'is_disabled' => 0,
                    'department_email' => $defaultEmail,
                    'soft_deleted' => 0,
                ]);
            }

            DB::commit();

            $this->info('  ✓ Administrator account created successfully');
            $this->info('     Email: ' . $email);

            return true;
        } catch (\Throwable $e) {
            DB::rollBack();
            $this->error('  ✗ Failed to create administrator: ' . $e->getMessage());

            return false;
        }
    }
}
