<?php

/**
 * Importer Extension for Ticaga
 *
 * Import tickets, customers, and responses from:
 * - CSV files
 * - SupportPal database
 *
 * Features:
 * - Field mapping
 * - Data preview
 * - Batch processing
 * - Progress tracking
 * - Error handling
 */

namespace App\Extensions\Installed\Importer;

use App\Extensions\Extension;
use Illuminate\Support\Facades\Log;

class ImporterExtension extends Extension
{
    /**
     * Install the extension
     *
     * Creates necessary database tables for tracking imports
     */
    public function install(): void
    {
        parent::install(); // Run migrations

        Log::info("Importer Extension installed successfully!");
    }

    /**
     * Enable the extension
     *
     * Register any hooks or services when enabled
     */
    public function enable(): void
    {
        Log::info("Importer Extension enabled!");

        // The extension is primarily UI-based, so no hooks to register
        // All functionality is accessed through the /extensions/importer routes
    }

    /**
     * Disable the extension
     *
     * Clean up any temporary data
     */
    public function disable(): void
    {
        Log::info("Importer Extension disabled!");

        // Clean up any temporary import files
        $this->cleanupTemporaryFiles();
    }

    /**
     * Uninstall the extension
     *
     * Remove all extension data
     */
    public function uninstall(): void
    {
        Log::info("Importer Extension uninstalled!");

        // Clean up temporary files
        $this->cleanupTemporaryFiles();

        // Note: Database tables are not automatically dropped
        // Users can manually drop tables if needed:
        // - importer_jobs
        // - importer_mappings
    }

    /**
     * Upgrade the extension
     *
     * @param string $fromVersion
     * @param string $toVersion
     */
    public function upgrade(string $fromVersion, string $toVersion): void
    {
        Log::info("Importer Extension upgrading from {$fromVersion} to {$toVersion}");

        // Add version-specific upgrade logic here if needed
    }

    /**
     * Register views and Livewire components
     */
    public function registerViews(): void
    {
        // Call parent to register view namespace
        parent::registerViews();

        // Register Livewire components
        \Livewire\Livewire::component('importer.index', \App\Extensions\Installed\Importer\Livewire\Index::class);
        \Livewire\Livewire::component('importer.csv.upload', \App\Extensions\Installed\Importer\Livewire\Csv\Upload::class);
        \Livewire\Livewire::component('importer.csv.mapping', \App\Extensions\Installed\Importer\Livewire\Csv\Mapping::class);
        \Livewire\Livewire::component('importer.csv.process', \App\Extensions\Installed\Importer\Livewire\Csv\Process::class);
        \Livewire\Livewire::component('importer.jobs.index', \App\Extensions\Installed\Importer\Livewire\Jobs\JobsIndex::class);
        \Livewire\Livewire::component('importer.database.configure', \App\Extensions\Installed\Importer\Livewire\Database\Configure::class);
        \Livewire\Livewire::component('importer.database.process', \App\Extensions\Installed\Importer\Livewire\Database\Process::class);
        \Livewire\Livewire::component('importer.api.configure', \App\Extensions\Installed\Importer\Livewire\Api\Configure::class);
        \Livewire\Livewire::component('importer.api.process', \App\Extensions\Installed\Importer\Livewire\Api\Process::class);
    }

    /**
     * Clean up temporary import files
     *
     * @return void
     */
    protected function cleanupTemporaryFiles(): void
    {
        $tempPath = storage_path('app/importer/temp');

        if (file_exists($tempPath)) {
            $files = glob($tempPath . '/*');
            foreach ($files as $file) {
                if (is_file($file) && filemtime($file) < strtotime('-24 hours')) {
                    @unlink($file);
                }
            }
        }
    }

    /**
     * ========================================
     * CONFIGURATION & SETTINGS HELPERS
     * ========================================
     */

    /**
     * Get extension settings from the database.
     *
     * HOW TO USE SETTINGS IN YOUR EXTENSION:
     *
     * Settings are configured by users via the Extensions UI (Configure button).
     * They are stored as key-value pairs in the database.
     *
     * Example: Get all settings
     * $settings = $this->getExtensionSettings();
     *
     * Example: Get specific setting with fallback
     * $apiKey = $settings['api_key'] ?? 'default-key';
     *
     * Example: Check if setting exists
     * if (isset($settings['csv_delimiter'])) {
     *     $delimiter = $settings['csv_delimiter'];
     * }
     *
     * COMMON IMPORTER SETTINGS:
     * - csv_delimiter: Comma, semicolon, tab, etc. (default: ,)
     * - batch_size: Number of records to import per batch (default: 100)
     * - max_file_size: Maximum CSV file size in MB (default: 10)
     * - supportpal_host: SupportPal database host
     * - supportpal_database: SupportPal database name
     * - supportpal_username: SupportPal database username
     * - supportpal_password: SupportPal database password
     *
     * @return array Array of key-value settings (e.g., ['csv_delimiter' => ',', 'batch_size' => '100'])
     */
    protected function getExtensionSettings(): array
    {
        // Access the extension settings from the database
        // These are the settings users configure via the Extensions UI
        $extension = \App\Models\Extension::where('extension_id', $this->getId())->first();

        if (!$extension || empty($extension->settings)) {
            return [];
        }

        return json_decode($extension->settings, true) ?? [];
    }

    /**
     * Save extension settings to the database.
     *
     * NOTE: Normally you don't need to call this directly.
     * Users configure settings via the Extensions UI.
     *
     * However, you might use this to:
     * - Set default settings during install() (e.g., default batch_size)
     * - Update settings programmatically based on events
     * - Store last import timestamp or statistics
     *
     * Example usage:
     * $this->saveExtensionSettings([
     *     'last_import' => now()->toDateTimeString(),
     *     'total_imported' => 1234,
     * ]);
     *
     * @param array $settings Key-value pairs (e.g., ['batch_size' => '100'])
     * @return void
     */
    protected function saveExtensionSettings(array $settings): void
    {
        $extension = \App\Models\Extension::where('extension_id', $this->getId())->first();

        if ($extension) {
            $extension->settings = json_encode($settings);
            $extension->save();
        }
    }

    /**
     * Get a specific setting with a default fallback.
     *
     * Convenience method for getting single settings.
     *
     * Example:
     * $batchSize = $this->getSetting('batch_size', 100);
     * $delimiter = $this->getSetting('csv_delimiter', ',');
     *
     * @param string $key Setting key
     * @param mixed $default Default value if setting doesn't exist
     * @return mixed
     */
    protected function getSetting(string $key, $default = null)
    {
        $settings = $this->getExtensionSettings();
        return $settings[$key] ?? $default;
    }
}
