<?php

namespace App\Extensions\Installed\Importer\Models;

use Carbon\Carbon;
use Carbon\CarbonInterval;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Import Job Model
 *
 * Tracks import progress and allows resuming interrupted imports
 */
class ImportJob extends Model
{
    protected $table = 'importer_jobs';

    protected $fillable = [
        'name',
        'source',
        'type',
        'status',
        'total_rows',
        'processed_rows',
        'failed_rows',
        'current_offset',
        'chunk_size',
        'file_path',
        'db_config',
        'field_mapping',
        'options',
        'errors',
        'result_summary',
        'started_at',
        'completed_at',
    ];

    protected $casts = [
        'db_config' => 'array',
        'field_mapping' => 'array',
        'options' => 'array',
        'errors' => 'array',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    /**
     * Mapping relation records.
     */
    public function mappings(): HasMany
    {
        return $this->hasMany(ImportMapping::class, 'job_id');
    }

    /**
     * Start the import job
     */
    public function start(): void
    {
        $this->update([
            'status' => 'processing',
            'started_at' => now(),
            'processed_rows' => 0,
            'failed_rows' => 0,
            'current_offset' => 0,
            'errors' => [],
        ]);
    }

    /**
     * Update progress after processing a chunk
     *
     * @param int $processedCount Rows successfully processed in this chunk
     * @param int $failedCount Rows failed in this chunk
     * @param array $errors Errors from this chunk
     */
    public function updateProgress(int $processedCount, int $failedCount = 0, array $errors = []): void
    {
        $this->increment('processed_rows', $processedCount);
        $this->increment('failed_rows', $failedCount);
        $this->increment('current_offset', $this->chunk_size);

        // Append errors
        if (!empty($errors)) {
            $allErrors = $this->errors ?? [];
            $this->update(['errors' => array_merge($allErrors, $errors)]);
        }
    }

    /**
     * Mark job as completed
     *
     * @param string|null $summary
     */
    public function complete(?string $summary = null): void
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
            'result_summary' => $summary ?? $this->getDefaultSummary(),
        ]);
    }

    /**
     * Mark job as failed
     *
     * @param string $reason
     */
    public function fail(string $reason): void
    {
        $this->update([
            'status' => 'failed',
            'completed_at' => now(),
            'result_summary' => "Import failed: {$reason}",
        ]);
    }

    /**
     * Pause the import job
     */
    public function pause(): void
    {
        $this->update(['status' => 'paused']);
    }

    /**
     * Resume the import job
     */
    public function resume(): void
    {
        $this->update(['status' => 'processing']);
    }

    /**
     * Check if there's more data to process
     *
     * @return bool
     */
    public function hasMoreToProcess(): bool
    {
        return $this->current_offset < $this->total_rows;
    }

    /**
     * Get progress percentage
     *
     * @return float
     */
    public function getProgressPercentage(): float
    {
        if ($this->total_rows === 0) {
            return 0;
        }

        return round(($this->processed_rows / $this->total_rows) * 100, 2);
    }

    /**
     * Estimate remaining processing time in seconds.
     */
    public function estimateTimeRemaining(): ?int
    {
        if (
            $this->processed_rows <= 0 ||
            $this->total_rows <= $this->processed_rows ||
            !$this->started_at
        ) {
            return null;
        }

        $referenceTime = $this->updated_at ?? Carbon::now();
        $elapsed = $this->started_at->diffInSeconds($referenceTime);

        if ($elapsed <= 0) {
            return null;
        }

        $avgPerRow = $elapsed / max($this->processed_rows, 1);
        if ($avgPerRow <= 0) {
            return null;
        }

        $remainingRows = max($this->total_rows - $this->processed_rows, 0);

        return (int) round($avgPerRow * $remainingRows);
    }

    /**
     * Human-readable ETA string.
     */
    public function etaForHumans(): ?string
    {
        $seconds = $this->estimateTimeRemaining();

        if ($seconds === null) {
            return null;
        }

        return CarbonInterval::seconds($seconds)
            ->cascade()
            ->forHumans([
                'parts' => 2,
                'short' => true,
            ]);
    }

    /**
     * Get default summary message
     *
     * @return string
     */
    protected function getDefaultSummary(): string
    {
        return sprintf(
            "Imported %d of %d %s. %d failed.",
            $this->processed_rows,
            $this->total_rows,
            $this->type,
            $this->failed_rows
        );
    }

    /**
     * Scope: Get active imports
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['processing', 'paused']);
    }

    /**
     * Scope: Get completed imports
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope: Get failed imports
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope: Get imports by type
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope: Get imports by source
     */
    public function scopeFromSource($query, string $source)
    {
        return $query->where('source', $source);
    }
}
