<?php

namespace App\Extensions\Installed\Importer\Services\DatabaseAdapters;

/**
 * WHMCS database adapter.
 *
 * Handles schema quirks for pulling data straight from a WHMCS installation.
 */
class WhmcsAdapter extends AbstractDatabaseImportAdapter
{
    protected const DEFAULT_CATEGORY_ID = 1;
    protected const DEFAULT_PREFIX_CANDIDATES = ['tbl', 'tbl_'];

    /**
     * Resolved table names keyed by logical stage.
     *
     * @var array<string,string|null>
     */
    protected array $tables = [];

    protected ?array $operatorIds = null;
    protected array $adminUsernameMap = [];
    protected array $contactUserCache = [];
    protected array $emailUserCache = [];

    public function boot(array $dbConfig, array $options = []): void
    {
        parent::boot($dbConfig, $options);

        $this->autoDetectPrefix();
        $this->initialiseTableMap();
    }

    public static function id(): string
    {
        return 'whmcs';
    }

    public static function label(): string
    {
        return 'WHMCS';
    }

    public static function defaults(): array
    {
        return [
            'chunk_size' => 100,
            'options' => [
                'import_departments' => true,
                'import_categories' => true,
                'import_users' => true,
                'import_operators' => true,
                'import_tickets' => true,
                'import_responses' => true,
                'import_articles' => true,
            ],
        ];
    }

    public static function isPreview(): bool
    {
        return false;
    }

    public function stages(): array
    {
        return [
            'departments',
            'categories',
            'users',
            'operators',
            'tickets',
            'responses',
            'articles',
        ];
    }

    public function count(string $stage): int
    {
        return match ($stage) {
            'departments' => $this->countSimple('departments'),
            'categories' => $this->countCategories(),
            'users' => $this->countSimple('users'),
            'operators' => $this->countSimple('operators'),
            'tickets' => $this->countSimple('tickets'),
            'responses' => $this->countSimple('responses'),
            'articles' => $this->countSimple('articles'),
            default => 0,
        };
    }

    public function fetch(string $stage, int $offset, int $limit): array
    {
        return match ($stage) {
            'departments' => $this->fetchDepartments($offset, $limit),
            'categories' => $this->fetchCategories($offset, $limit),
            'users' => $this->fetchUsers($offset, $limit),
            'operators' => $this->fetchOperators($offset, $limit),
            'tickets' => $this->fetchTickets($offset, $limit),
            'responses' => $this->fetchResponses($offset, $limit),
            'articles' => $this->fetchArticles($offset, $limit),
            default => [],
        };
    }

    public function mapStatus(?int $statusId): string
    {
        return match ($statusId) {
            2 => 'closed',
            3 => 'awaiting reply',
            4 => 'in progress',
            default => 'open',
        };
    }

    public function mapPriority(?int $priorityId): string
    {
        return match ($priorityId) {
            1 => 'low',
            2 => 'medium',
            3 => 'high',
            4 => 'emergency',
            default => 'none',
        };
    }

    public function isOperatorUser(int $userId): bool
    {
        if ($this->operatorIds === null) {
            $table = $this->resolvedTable('operators');
            if (!$table || !$this->tableExists($table)) {
                $this->operatorIds = [];
            } else {
                $stmt = $this->connection()->query("
                    SELECT id
                      FROM {$this->table($table)}
                ");
                $this->operatorIds = $stmt ? array_map('intval', array_column($stmt->fetchAll(), 'id')) : [];
            }
        }

        return in_array($userId, $this->operatorIds ?? [], true);
    }

    public function findUser(int $userId): ?array
    {
        if ($userId <= 0) {
            return null;
        }

        $clientTable = $this->resolvedTable('users');
        if ($clientTable && $this->tableExists($clientTable)) {
            $stmt = $this->connection()->prepare("
                SELECT c.*,
                       c.companyname AS company,
                       c.companyname AS organisation_name,
                       c.datecreated AS created_at,
                       c.lastlogin AS updated_at,
                       " . ($this->columnExists($clientTable, 'emailverified') ? 'c.emailverified' : '1') . " AS confirmed
                  FROM {$this->table($clientTable)} c
                 WHERE c.id = :id
                 LIMIT 1
            ");
            $stmt->bindValue(':id', $userId, \PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch();
            if ($row) {
                return $row;
            }
        }

        $operatorTable = $this->resolvedTable('operators');
        if ($operatorTable && $this->tableExists($operatorTable)) {
            $stmt = $this->connection()->prepare("
                SELECT a.*,
                       a.id AS user_id,
                       a.email AS user_email,
                       a.firstname,
                       a.lastname,
                       1 AS confirmed
                  FROM {$this->table($operatorTable)} a
                 WHERE a.id = :id
                 LIMIT 1
            ");
            $stmt->bindValue(':id', $userId, \PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch();
            if ($row) {
                return $row;
            }
        }

        return null;
    }

    protected function autoDetectPrefix(): void
    {
        if ($this->tableExists('ticketdepartments')) {
            return;
        }

        foreach (self::DEFAULT_PREFIX_CANDIDATES as $candidate) {
            if ($this->rawTableExists($candidate . 'ticketdepartments')) {
                $this->dbConfig['prefix'] = $candidate;
                $this->prefix = $candidate;
                return;
            }
        }
    }

    protected function initialiseTableMap(): void
    {
        $this->tables = [
            'departments' => $this->detectTable(['ticketdepartments', 'supportdepartments']),
            'categories' => $this->detectTable(['announcementcats', 'knowledgebasecats']),
            'articles' => $this->detectTable(['announcements', 'knowledgebase']),
            'users' => $this->detectTable(['clients', 'users']),
            'contacts' => $this->detectTable(['contacts']),
            'operators' => $this->detectTable(['admins', 'staff']),
            'tickets' => $this->detectTable(['tickets']),
            'responses' => $this->detectTable(['ticketreplies']),
            'ticketnotes' => $this->detectTable(['ticketnotes']),
            'ticketcc' => $this->detectTable(['ticketcc']),
            'ticket_history' => $this->detectTable(['tickethistory', 'ticketlog']),
        ];
    }

    protected function detectTable(array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if ($this->tableExists($candidate)) {
                return $candidate;
            }
        }

        return null;
    }

    protected function resolvedTable(string $key): ?string
    {
        return $this->tables[$key] ?? null;
    }

    protected function countSimple(string $key): int
    {
        $table = $this->resolvedTable($key);
        if (!$table || !$this->tableExists($table)) {
            return 0;
        }

        $stmt = $this->connection()->query("
            SELECT COUNT(*)
              FROM {$this->table($table)}
        ");

        return (int) $stmt->fetchColumn();
    }

    protected function countCategories(): int
    {
        $table = $this->resolvedTable('categories');
        if ($table && $this->tableExists($table)) {
            return $this->countSimple('categories');
        }

        // If announcements exist but categories table is missing, synthesize a default category.
        $articlesTable = $this->resolvedTable('articles');
        if ($articlesTable && $this->tableExists($articlesTable)) {
            return 1;
        }

        return 0;
    }

    protected function fetchDepartments(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('departments');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT *
              FROM {$this->table($table)}
             ORDER BY id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    protected function fetchCategories(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('categories');
        if ($table && $this->tableExists($table)) {
            $stmt = $this->connection()->prepare("
                SELECT *
                  FROM {$this->table($table)}
                 ORDER BY id ASC
                 LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
            $stmt->execute();

            $rows = $stmt->fetchAll();

            return array_map(function (array $row): array {
                if (!isset($row['status']) && isset($row['hidden'])) {
                    $row['status'] = $row['hidden'] ? 0 : 1;
                }

                return $row;
            }, $rows);
        }

        if ($offset > 0) {
            return [];
        }

        return [[
            'id' => self::DEFAULT_CATEGORY_ID,
            'name' => 'General',
            'status' => 1,
        ]];
    }

    protected function fetchUsers(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('users');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $select = [
            'c.*',
        ];

        if ($this->columnExists($table, 'companyname')) {
            $select[] = 'c.companyname AS company';
            $select[] = 'c.companyname AS organisation_name';
        }

        if ($this->columnExists($table, 'emailverified')) {
            $select[] = 'c.emailverified AS confirmed';
        } else {
            $select[] = '1 AS confirmed';
        }

        if ($this->columnExists($table, 'datecreated')) {
            $select[] = 'c.datecreated AS created_at';
        }

        if ($this->columnExists($table, 'lastlogin')) {
            $select[] = 'c.lastlogin AS updated_at';
        }

        $sql = sprintf(
            'SELECT %s FROM %s ORDER BY c.id ASC LIMIT :limit OFFSET :offset',
            implode(', ', $select),
            $this->table($table) . ' c'
        );

        $stmt = $this->connection()->prepare($sql);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    protected function fetchOperators(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('operators');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $select = [
            'a.*',
            'a.id AS user_id',
            'a.email AS user_email',
        ];

        $sql = sprintf(
            'SELECT %s FROM %s ORDER BY a.id ASC LIMIT :limit OFFSET :offset',
            implode(', ', $select),
            $this->table($table) . ' a'
        );

        $stmt = $this->connection()->prepare($sql);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    protected function fetchTickets(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('tickets');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $ccTable = $this->resolvedTable('ticketcc');

        if ($ccTable && $this->tableExists($ccTable)) {
            $query = "
                SELECT t.*, GROUP_CONCAT(DISTINCT cc.cc) AS cc_emails
                  FROM {$this->table($table)} t
                  LEFT JOIN {$this->table($ccTable)} cc
                    ON cc.ticketid = t.id
                 GROUP BY t.id
                 ORDER BY t.id ASC
                 LIMIT :limit OFFSET :offset
            ";
        } else {
            $query = "
                SELECT t.*
                  FROM {$this->table($table)} t
                 ORDER BY t.id ASC
                 LIMIT :limit OFFSET :offset
            ";
        }

        $stmt = $this->connection()->prepare($query);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll();

        return array_map(function (array $row): array {
            $row['number'] = $row['number'] ?? ($row['tid'] ?? ($row['ticketnum'] ?? null));

            $departmentId = $row['deptid'] ?? $row['departmentid'] ?? null;
            if ($departmentId !== null) {
                $row['department_id'] = (int) $departmentId;
            }

            $userId = $row['userid'] ?? null;
            if (empty($userId) && !empty($row['contactid'])) {
                $userId = $this->resolveUserIdFromContact((int) $row['contactid']);
            }
            if (empty($userId) && !empty($row['email'])) {
                $userId = $this->resolveClientIdByEmail((string) $row['email']);
            }
            $row['user_id'] = $userId !== null ? (int) $userId : null;

            $row['status_label'] = $row['status'] ?? null;
            $row['status_id'] = $this->statusCodeFromValue($row['status'] ?? null);

            $priorityRaw = $row['priority'] ?? $row['urgency'] ?? null;
            $row['priority_label'] = $priorityRaw;
            $row['priority_id'] = $this->priorityCodeFromValue($priorityRaw);

            $row['subject'] = $row['subject'] ?? ($row['title'] ?? ($row['subject_line'] ?? "Ticket {$row['id']}"));

            $assigned = $row['flag'] ?? ($row['adminid'] ?? null);
            if ($assigned === null && !empty($row['admin'])) {
                $assigned = $this->adminIdByUsername((string) $row['admin']);
            }
            $row['assigned_to'] = $assigned !== null ? (int) $assigned : null;

            $row['created_at'] = $row['created_at'] ?? $row['date'] ?? null;
            $row['updated_at'] = $row['updated_at'] ?? $row['lastreply'] ?? $row['date'] ?? null;
            $row['last_reply_at'] = $row['last_reply_at'] ?? $row['lastreply'] ?? null;
            $row['first_staff_reply_at'] = $row['first_staff_reply_at'] ?? ($row['firstresponse'] ?? null);

            $row['resolved_at'] = $row['resolved_at'] ?? ($row['closed'] ?? null);
            $row['closed_at'] = $row['closed_at'] ?? ($row['closed'] ?? null);

            if (isset($row['cc_emails'])) {
                $row['cc_list'] = $this->parseCcList($row['cc_emails']);
                unset($row['cc_emails']);
            } else {
                $row['cc_list'] = [];
            }

            $row['assignment_history'] = $this->fetchTicketHistoryEntries((int) ($row['id'] ?? 0));

            return $row;
        }, $rows);
    }

    protected function fetchResponses(int $offset, int $limit): array
    {
        $responsesTable = $this->resolvedTable('responses');
        $ticketsTable = $this->resolvedTable('tickets');

        if (
            !$responsesTable
            || !$ticketsTable
            || !$this->tableExists($responsesTable)
            || !$this->tableExists($ticketsTable)
        ) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT r.*,
                   r.tid AS ticket_id,
                   r.date AS created_at,
                   r.date AS updated_at
              FROM {$this->table($responsesTable)} r
              INNER JOIN {$this->table($ticketsTable)} t
                ON t.id = r.tid
             ORDER BY r.date ASC, r.id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll();

        return array_map(function (array $row): array {
            $userId = $row['userid'] ?? null;
            if (empty($userId) && !empty($row['contactid'])) {
                $userId = $this->resolveUserIdFromContact((int) $row['contactid']);
            }
            if (empty($userId) && !empty($row['adminid'])) {
                $userId = (int) $row['adminid'];
            }
            if (empty($userId) && !empty($row['admin'])) {
                $userId = $this->adminIdByUsername((string) $row['admin']);
            }

            $row['user_id'] = $userId !== null ? (int) $userId : null;
            $row['purified_text'] = $row['purified_text'] ?? $row['message'] ?? '';
            $row['text'] = $row['text'] ?? $row['message'] ?? '';
            $row['ip_address'] = $row['ip_address'] ?? ($row['ip'] ?? null);
            $row['type'] = $row['type'] ?? 0;

            return $row;
        }, $rows);
    }

    protected function fetchArticles(int $offset, int $limit): array
    {
        $table = $this->resolvedTable('articles');
        if (!$table || !$this->tableExists($table)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT *
              FROM {$this->table($table)}
             ORDER BY id ASC
             LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll();

        return array_map(function (array $row): array {
            $category = $row['category'] ?? $row['catid'] ?? $row['category_id'] ?? null;
            if ($category === null) {
                $category = self::DEFAULT_CATEGORY_ID;
            }
            $row['category_id'] = (int) $category;

            $row['title'] = $row['title'] ?? ($row['name'] ?? null);
            $row['content'] = $row['content'] ?? ($row['article'] ?? ($row['announcement'] ?? ''));
            $row['slug'] = $row['slug'] ?? ($row['seolink'] ?? ($row['title'] ?? null));
            $row['status'] = isset($row['published']) ? (int) $row['published'] : ($row['status'] ?? 1);
            $row['visibility'] = isset($row['private']) ? (int) $row['private'] : ($row['visibility'] ?? 0);
            $row['tags'] = $row['tags'] ?? '';
            $row['views'] = $row['views'] ?? 0;
            $row['likes'] = $row['likes'] ?? ($row['usefulness'] ?? 0);

            $row['created_at'] = $row['created_at'] ?? ($row['date'] ?? null);
            $row['updated_at'] = $row['updated_at'] ?? ($row['date'] ?? null);

            return $row;
        }, $rows);
    }

    protected function statusCodeFromValue(mixed $value): int
    {
        if ($value === null) {
            return 1;
        }

        if (is_int($value)) {
            return $value;
        }

        $normalized = strtolower(trim((string) $value));

        return match ($normalized) {
            'closed', 'resolved' => 2,
            'answered', 'customer-reply', 'customer reply', 'pending client' => 3,
            'in progress', 'on hold', 'pending', 'escalated' => 4,
            default => 1,
        };
    }

    protected function priorityCodeFromValue(mixed $value): int
    {
        if ($value === null) {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        $normalized = strtolower(trim((string) $value));

        return match ($normalized) {
            'low' => 1,
            'medium', 'normal' => 2,
            'high' => 3,
            'critical', 'emergency', 'urgent' => 4,
            default => 0,
        };
    }

    protected function adminIdByUsername(?string $username): ?int
    {
        if ($username === null || $username === '') {
            return null;
        }

        if (array_key_exists($username, $this->adminUsernameMap)) {
            return $this->adminUsernameMap[$username];
        }

        $table = $this->resolvedTable('operators');
        if (!$table || !$this->tableExists($table) || !$this->columnExists($table, 'username')) {
            $this->adminUsernameMap[$username] = null;

            return null;
        }

        $stmt = $this->connection()->prepare("
            SELECT id
              FROM {$this->table($table)}
             WHERE username = :username
             LIMIT 1
        ");
        $stmt->bindValue(':username', $username);
        $stmt->execute();
        $id = $stmt->fetchColumn();

        $this->adminUsernameMap[$username] = $id !== false ? (int) $id : null;

        return $this->adminUsernameMap[$username];
    }

    protected function resolveUserIdFromContact(int $contactId): ?int
    {
        if ($contactId <= 0) {
            return null;
        }

        if (array_key_exists($contactId, $this->contactUserCache)) {
            return $this->contactUserCache[$contactId];
        }

        $table = $this->resolvedTable('contacts');
        if (!$table || !$this->tableExists($table)) {
            $this->contactUserCache[$contactId] = null;

            return null;
        }

        $stmt = $this->connection()->prepare("
            SELECT userid
              FROM {$this->table($table)}
             WHERE id = :id
             LIMIT 1
        ");
        $stmt->bindValue(':id', $contactId, \PDO::PARAM_INT);
        $stmt->execute();
        $userId = $stmt->fetchColumn();

        $this->contactUserCache[$contactId] = $userId !== false ? (int) $userId : null;

        return $this->contactUserCache[$contactId];
    }

    protected function resolveClientIdByEmail(string $email): ?int
    {
        $normalized = strtolower(trim($email));
        if ($normalized === '') {
            return null;
        }

        if (array_key_exists($normalized, $this->emailUserCache)) {
            return $this->emailUserCache[$normalized];
        }

        $table = $this->resolvedTable('users');
        if (!$table || !$this->tableExists($table)) {
            $this->emailUserCache[$normalized] = null;

            return null;
        }

        $stmt = $this->connection()->prepare("
            SELECT id
              FROM {$this->table($table)}
             WHERE LOWER(email) = :email
             LIMIT 1
        ");
        $stmt->bindValue(':email', $normalized);
        $stmt->execute();

        $id = $stmt->fetchColumn();
        $this->emailUserCache[$normalized] = $id !== false ? (int) $id : null;

        return $this->emailUserCache[$normalized];
    }

    protected function rawTableExists(string $tableName): bool
    {
        $stmt = $this->connection()->prepare("
            SELECT COUNT(*)
              FROM information_schema.TABLES
             WHERE TABLE_SCHEMA = DATABASE()
               AND TABLE_NAME = :table
        ");
        $stmt->bindValue(':table', $tableName);
        $stmt->execute();

        return (bool) $stmt->fetchColumn();
    }

    /**
     * Parse CC string into a normalized email list.
     *
     * @param string|null $value
     * @return array<int,string>
     */
    protected function parseCcList(?string $value): array
    {
        if ($value === null || trim($value) === '') {
            return [];
        }

        $parts = preg_split('/[,\r\n]+/', $value) ?: [];

        $emails = array_values(array_filter(array_unique(array_map(
            static fn ($email) => strtolower(trim($email)),
            $parts
        )), static fn ($email) => filter_var($email, FILTER_VALIDATE_EMAIL)));

        return $emails;
    }

    /**
     * Fetch ticket history entries (assignment logs, etc.).
     *
     * @return array<int,array<string,mixed>>
     */
    protected function fetchTicketHistoryEntries(int $ticketId): array
    {
        if ($ticketId <= 0) {
            return [];
        }

        $historyTable = $this->resolvedTable('ticket_history');
        if (!$historyTable || !$this->tableExists($historyTable)) {
            return [];
        }

        $stmt = $this->connection()->prepare("
            SELECT *
              FROM {$this->table($historyTable)}
             WHERE ticketid = :ticket
             ORDER BY date ASC
        ");
        $stmt->bindValue(':ticket', $ticketId, \PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll();
        if (!$rows) {
            return [];
        }

        return array_map(static function (array $row): array {
            return [
                'timestamp' => $row['date'] ?? $row['created_at'] ?? null,
                'author' => $row['admin'] ?? $row['name'] ?? $row['username'] ?? null,
                'action' => $row['title'] ?? $row['event'] ?? null,
                'message' => $row['description'] ?? $row['message'] ?? $row['details'] ?? null,
            ];
        }, $rows);
    }
}
