<?php

namespace App\Http\Controllers;

use App\Models\{User, Tickets};

use Illuminate\Auth\Events\Registered;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\{Auth, Hash, DB};
use Illuminate\Validation\Rules\Password;

class ClientsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (!auth()->user()->can('view customers')) {
            return redirect()->route('dashboard');
        }
        return view('clients.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create() : void
    {
        $validated = $this->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'password' => ['required', 'string', 'confirmed', Password::min(15)->uncompromised()->letters()->numbers()->symbols()],
        ]);

        $validated['password'] = Hash::make($validated['password']);

        event(new Registered(($user = User::create($validated)->assignRole('customer'))));
        Auth::login($user);
        $this->redirectIntended(route('dashboard', absolute: false), navigate: true);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request)
    {
        $client_id = $request->route('clientId');

        if (!auth()->user()->can('view customers')) {
            if($client_id == Auth::user()->id)
            {
                return redirect()->route('settings.profile');
            } else {
                return redirect()->route('dashboard');
            }
        }

        if($client_id == Auth::user()->id)
        {
            return redirect()->route('settings.profile');
        } else {
            $user = User::where('id', '=', $client_id)->first();
            $account_managers = User::all();
            if($user)
            {
                $total_open = Tickets::where('user_id', '=', $client_id)->where('status', '!=', 'closed')->count();
                $total_closed = Tickets::where('user_id', '=', $client_id)->where('status', '=', 'closed')->count();
                $user_roles = implode($user->roles()->get()->pluck('name', 'id')->toArray());
                
                return view('clients.view', [
                    'total_open' => $total_open,
                    'total_closed' => $total_closed,
                    'user' => $user,
                    'account_managers' => $account_managers,
                    'user_roles' => $user_roles,
                ]);
            } else {
                return view('clients.index');
            }
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function assigned()
    {
        if (!auth()->user()->can('view assigned customers')) {
            return redirect()->route('dashboard');
        }

        return view('clients.assigned');
    }
}
