<?php

namespace App\Http\Controllers;

use App\Models\Departments;
use App\Models\User;
use App\Services\Installation\InstallationService;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules\Password;

class InstallController extends Controller
{
    protected array $steps = [
        'requirements',
        'dependencies',
        'database',
        'application',
        'administrator',
    ];

    public function __construct(protected InstallationService $installation)
    {
    }

    public function index(): RedirectResponse
    {
        if (! $this->installation->needsInstallation()) {
            return redirect()->route('dashboard');
        }

        return redirect()->route('install.show', ['step' => 'requirements']);
    }

    public function show(string $step): View|RedirectResponse
    {
        if (! in_array($step, $this->steps, true)) {
            abort(404);
        }

        if (! $this->installation->needsInstallation()) {
            return redirect()->route('dashboard');
        }

        $requirements = $this->installation->systemRequirements();
        $requirementsPassed = $this->installation->requirementsSatisfied($requirements);

        $dependenciesVerified = session('install.dependencies_verified', false);
        $shouldCheckDependencies = $dependenciesVerified || $step === 'dependencies';

        if ($shouldCheckDependencies) {
            $dependencies = $this->installation->dependencyStatus();
            $dependenciesPassed = $this->installation->dependenciesSatisfied($dependencies);

            if ($dependenciesVerified && ! $dependenciesPassed) {
                session()->forget('install.dependencies_verified');
                $dependenciesVerified = false;
            }
        } else {
            $dependencies = $this->installation->pendingDependencyStatus();
            $dependenciesPassed = false;
        }

        $shouldCheckDatabase = in_array($step, ['database', 'application', 'administrator'], true);

        if ($shouldCheckDatabase) {
            $connected = $this->installation->databaseIsReachable();

            $connectionStatus = [
                'connected' => $connected,
                'hasMigrations' => $connected ? $this->installation->hasMigrationsTable() : false,
                'hasUsersTable' => $connected ? $this->installation->hasUsersTable() : false,
                'hasUsers' => $connected ? $this->installation->hasUsers() : false,
                'error' => $this->installation->getConnectionError(),
            ];
        } else {
            $connectionStatus = [
                'connected' => false,
                'hasMigrations' => false,
                'hasUsersTable' => false,
                'hasUsers' => false,
                'error' => null,
            ];
        }

        $databaseVerified = session('install.database_verified', false);

        if ($databaseVerified && ! $connectionStatus['connected']) {
            session()->forget('install.database_verified');
            $databaseVerified = false;
        }

        $applicationConfigured = session()->has('install.application');

        if ($redirect = $this->guardStepAccess($step, $requirementsPassed, $dependenciesPassed)) {
            return $redirect;
        }

        $defaults = [
            'db_host' => config('database.connections.mysql.host'),
            'db_port' => config('database.connections.mysql.port'),
            'db_name' => config('database.connections.mysql.database'),
            'db_username' => config('database.connections.mysql.username'),
            'app_url' => config('app.url'),
        ];

        $databaseSession = session('install.database', []);
        $applicationSession = session('install.application', []);

        $databaseData = [
            'db_host' => $databaseSession['host'] ?? $defaults['db_host'],
            'db_port' => $databaseSession['port'] ?? $defaults['db_port'],
            'db_name' => $databaseSession['database'] ?? $defaults['db_name'],
            'db_username' => $databaseSession['username'] ?? $defaults['db_username'],
        ];

        $applicationData = [
            'app_url' => $applicationSession['app_url'] ?? $defaults['app_url'],
        ];

        $wizardSteps = $this->buildWizardSteps(
            $step,
            $requirementsPassed,
            $dependenciesPassed,
            $databaseVerified,
            $applicationConfigured,
            $connectionStatus['hasUsers'],
        );

        return view('install.index', [
            'currentStep' => $step,
            'steps' => $wizardSteps,
            'requirements' => $requirements,
            'dependencies' => $dependencies,
            'connectionStatus' => $connectionStatus,
            'databaseData' => $databaseData,
            'applicationData' => $applicationData,
        ]);
    }

    public function handle(Request $request, string $step): RedirectResponse
    {
        if (! in_array($step, $this->steps, true)) {
            abort(404);
        }

        return match ($step) {
            'requirements' => $this->advanceFromRequirements(),
            'dependencies' => $this->handleDependencies($request),
            'database' => $this->handleDatabase($request),
            'application' => $this->handleApplication($request),
            'administrator' => $this->handleAdministrator($request),
            default => abort(404),
        };
    }

    protected function advanceFromRequirements(): RedirectResponse
    {
        if (! $this->installation->requirementsSatisfied()) {
            return redirect()->route('install.show', ['step' => 'requirements'])
                ->withErrors(['requirements' => __('Please ensure all system requirements are met before continuing.')]);
        }

        return redirect()->route('install.show', ['step' => 'dependencies']);
    }

    protected function handleDependencies(Request $request): RedirectResponse
    {
        if ($request->input('action') === 'install') {
            try {
                $this->installation->ensureDependenciesInstalled();
            } catch (\Throwable $exception) {
                return redirect()->route('install.show', ['step' => 'dependencies'])
                    ->withErrors(['dependencies' => __('Unable to install application dependencies automatically. :message', ['message' => $exception->getMessage()])]);
            }

            return redirect()->route('install.show', ['step' => 'dependencies'])
                ->with('status', __('Dependency installation completed.'));
        }

        $dependencies = $this->installation->dependencyStatus();

        if (! $this->installation->dependenciesSatisfied($dependencies)) {
            return redirect()->route('install.show', ['step' => 'dependencies'])
                ->withErrors(['dependencies' => __('Please install the missing dependencies before continuing.')]);
        }

        session()->put('install.dependencies_verified', true);

        return redirect()->route('install.show', ['step' => 'database']);
    }

    protected function handleDatabase(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'db_host' => ['required', 'string', 'max:255'],
            'db_port' => ['required', 'numeric', 'min:1', 'max:65535'],
            'db_name' => ['required', 'string', 'max:255'],
            'db_username' => ['required', 'string', 'max:255'],
            'db_password' => ['nullable', 'string', 'max:255'],
        ]);

        $databaseConfiguration = [
            'host' => $validated['db_host'],
            'port' => $validated['db_port'],
            'database' => $validated['db_name'],
            'username' => $validated['db_username'],
            'password' => $validated['db_password'] ?? '',
        ];

        $existingConfiguration = session('install.database');

        // Preserve password from session if not provided in form
        if (($validated['db_password'] ?? '') === '' && is_array($existingConfiguration) && array_key_exists('password', $existingConfiguration)) {
            $databaseConfiguration['password'] = $existingConfiguration['password'];
        }

        // Fall back to .env file password if still empty
        if ($databaseConfiguration['password'] === '') {
            // Try config first
            $envPassword = config('database.connections.mysql.password');

            // If config is empty, read directly from .env file
            if (! $envPassword) {
                $envPath = base_path('.env');
                if (file_exists($envPath)) {
                    $envContent = file_get_contents($envPath);
                    if (preg_match('/^DB_PASSWORD=(.*)$/m', $envContent, $matches)) {
                        $envPassword = trim($matches[1]);
                        // Remove quotes if present
                        $envPassword = trim($envPassword, '"\'');
                    }
                }
            }

            if ($envPassword) {
                $databaseConfiguration['password'] = $envPassword;
            }
        }

        try {
            $this->installation->testDatabaseConnection($databaseConfiguration);
        } catch (\Throwable $exception) {
            return back()
                ->withErrors(['database' => __('Unable to connect to the provided MySQL database. :message', ['message' => $exception->getMessage()])])
                ->withInput($request->except('db_password'));
        }

        session()->put('install.database', $databaseConfiguration);
        session()->put('install.database_verified', true);

        return redirect()->route('install.show', ['step' => 'application']);
    }

    protected function handleApplication(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'app_url' => ['required', 'url', 'max:255'],
        ]);

        // Automatically request trial license
        try {
            $trialLicenseKey = Controller::trial();

            if (empty($trialLicenseKey)) {
                return back()
                    ->withErrors(['application' => __('Unable to automatically request a trial license. Please check your internet connection and firewall settings.')])
                    ->withInput();
            }

            $validated['license_key'] = $trialLicenseKey;
        } catch (\Throwable $exception) {
            return back()
                ->withErrors(['application' => __('Failed to request trial license: :message', ['message' => $exception->getMessage()])])
                ->withInput();
        }

        session()->put('install.application', $validated);

        return redirect()->route('install.show', ['step' => 'administrator'])
            ->with('status', __('Trial license successfully requested and activated.'));
    }

    protected function handleAdministrator(Request $request): RedirectResponse
    {
        $databaseConfiguration = session('install.database');
        $applicationConfiguration = session('install.application');

        if (! is_array($databaseConfiguration) || ! is_array($applicationConfiguration)) {
            return redirect()->route('install.show', ['step' => 'requirements'])
                ->withErrors(['requirements' => __('Your installation session has expired. Please start again.')]);
        }

        $emailRules = ['required', 'string', 'lowercase', 'email', 'max:255'];

        if ($this->installation->hasUsersTable()) {
            $emailRules[] = 'unique:' . User::class;
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => $emailRules,
            'company' => ['nullable', 'string', 'max:255'],
            'phone_number' => ['nullable', 'string', 'max:255'],
            'password' => ['required', 'string', 'confirmed', Password::min(15)->uncompromised()->letters()->numbers()->symbols()],
        ]);

        if (! $this->installation->requirementsSatisfied()) {
            return redirect()->route('install.show', ['step' => 'requirements'])
                ->withErrors(['requirements' => __('Please ensure all system requirements are met before continuing.')]);
        }

        if (! $this->installation->dependenciesSatisfied()) {
            return redirect()->route('install.show', ['step' => 'dependencies'])
                ->withErrors(['dependencies' => __('Please install the missing dependencies before continuing.')]);
        }

        try {
            $this->installation->testDatabaseConnection($databaseConfiguration);
        } catch (\Throwable $exception) {
            return redirect()->route('install.show', ['step' => 'database'])
                ->withErrors(['database' => __('Unable to connect to the provided MySQL database. :message', ['message' => $exception->getMessage()])]);
        }

        // Store password before redacting from session
        $dbPassword = $databaseConfiguration['password'];

        $this->installation->updateEnvironmentFile([
            'APP_URL' => $applicationConfiguration['app_url'],
            'DB_CONNECTION' => 'mysql',
            'DB_HOST' => $databaseConfiguration['host'],
            'DB_PORT' => $databaseConfiguration['port'],
            'DB_DATABASE' => $databaseConfiguration['database'],
            'DB_USERNAME' => $databaseConfiguration['username'],
            'DB_PASSWORD' => $dbPassword,
            'LICENSE_KEY' => $applicationConfiguration['license_key'] ?? '',
            'APP_ENV' => 'production',
            'APP_DEBUG' => 'false',
            'CACHE_DRIVER' => 'database',
        ]);

        // Clear sensitive database password from session immediately after writing to .env
        $databaseConfiguration['password'] = '[REDACTED]';
        session()->put('install.database', $databaseConfiguration);

        config([
            'app.url' => $applicationConfiguration['app_url'],
            'database.default' => 'mysql',
            'database.connections.mysql.host' => $databaseConfiguration['host'],
            'database.connections.mysql.port' => $databaseConfiguration['port'],
            'database.connections.mysql.database' => $databaseConfiguration['database'],
            'database.connections.mysql.username' => $databaseConfiguration['username'],
            'database.connections.mysql.password' => $dbPassword,
            'ticaga.key' => $applicationConfiguration['license_key'] ?? '',
        ]);

        DB::purge('mysql');
        DB::reconnect('mysql');
        $this->installation->resetConnectionCheck();

        try {
            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
        } catch (\Throwable $exception) {
            Log::error('Installation failed while running migrations or seeders.', [
                'exception' => $exception,
                'database' => $databaseConfiguration['database'] ?? 'unknown',
            ]);

            // Provide helpful troubleshooting guidance
            $errorMessage = __('Database migration failed. Please check the following:') . "\n\n";
            $errorMessage .= '• ' . __('Database user has CREATE, ALTER, DROP, INSERT privileges') . "\n";
            $errorMessage .= '• ' . __('Database is empty or can be safely overwritten') . "\n";
            $errorMessage .= '• ' . __('MySQL/MariaDB server is running and accessible') . "\n\n";
            $errorMessage .= __('Technical error: :message', ['message' => $exception->getMessage()]);

            return redirect()->route('install.show', ['step' => 'database'])
                ->withErrors(['database' => $errorMessage]);
        }

        DB::beginTransaction();

        try {
            if (User::query()->where('email', $validated['email'])->exists()) {
                DB::rollBack();

                return redirect()->route('install.show', ['step' => 'administrator'])
                    ->withErrors(['administrator' => __('A user with this email address already exists.')])
                    ->withInput($request->except(['password', 'password_confirmation']));
            }

            $user = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'company' => $validated['company'] ?? '',
                'password' => Hash::make($validated['password']),
                'billing_id' => '0',
                'billing_system' => '',
                'account_manager' => '0',
                'biography' => '',
                'force_password_reset' => false,
            ]);

            $user->forceFill([
                'phone_number' => $validated['phone_number'] ?? '',
                'email_verified_at' => now(),
            ])->save();

            $user->assignRole('superadmin');

            $token = $user->createToken('Primary API Token')->plainTextToken;
            $user->update(['api_token' => $token]);

            if (! Departments::query()->exists()) {
                // Get domain from app URL for default email
                $appUrl = parse_url($applicationConfiguration['app_url'], PHP_URL_HOST) ?? 'localhost';
                $defaultEmail = 'noreply@' . $appUrl;

                Departments::create([
                    'department_name' => 'Support',
                    'department_description' => 'Default support department',
                    'slug' => Str::slug('support'),
                    'allows_high_priority' => 1,
                    'cc_enabled' => 1,
                    'is_public' => 1,
                    'is_disabled' => 0,
                    'department_email' => $defaultEmail,
                    'soft_deleted' => 0,
                ]);
            }

            DB::commit();
        } catch (\Throwable $exception) {
            DB::rollBack();
            Log::error('Installation failed while creating the initial user or seed data.', [
                'exception' => $exception,
                'email' => $validated['email'] ?? 'unknown',
            ]);

            // Provide helpful error message
            $errorMessage = __('Failed to create administrator account. ');
            $errorMessage .= __('The database has been rolled back to its previous state. ');
            $errorMessage .= __('Error: :message', ['message' => $exception->getMessage()]);

            return redirect()->route('install.show', ['step' => 'administrator'])
                ->withErrors(['administrator' => $errorMessage])
                ->withInput($request->except(['password', 'password_confirmation']));
        }

        $this->installation->markInstallationComplete();

        session()->forget('install');

        return redirect()->route('login')->with('status', __('Installation complete! You can now log in with your administrator account.'));
    }

    protected function guardStepAccess(string $step, bool $requirementsPassed, bool $dependenciesPassed): ?RedirectResponse
    {
        $databaseVerified = session('install.database_verified', false);
        $applicationConfigured = session()->has('install.application');

        return match ($step) {
            'dependencies' => $requirementsPassed
                ? null
                : redirect()->route('install.show', ['step' => 'requirements'])
                    ->withErrors(['requirements' => __('Please address the system requirements before continuing.')]),
            'database' => !$requirementsPassed
                ? redirect()->route('install.show', ['step' => 'requirements'])
                    ->withErrors(['requirements' => __('Please address the system requirements before continuing.')])
                : (!$dependenciesPassed
                    ? redirect()->route('install.show', ['step' => 'dependencies'])
                        ->withErrors(['dependencies' => __('Please install the missing dependencies before continuing.')])
                    : null),
            'application' => $databaseVerified
                ? null
                : redirect()->route('install.show', ['step' => 'database'])
                    ->withErrors(['database' => __('Please provide valid database details before continuing.')]),
            'administrator' => $databaseVerified && $applicationConfigured
                ? null
                : redirect()->route('install.show', ['step' => $databaseVerified ? 'application' : 'database'])
                    ->withErrors(['administrator' => __('Complete the earlier steps before creating the administrator account.')]),
            default => null,
        };
    }

    protected function buildWizardSteps(
        string $currentStep,
        bool $requirementsPassed,
        bool $dependenciesPassed,
        bool $databaseVerified,
        bool $applicationConfigured,
        bool $administratorCreated,
    ): array
    {
        return [
            [
                'key' => 'requirements',
                'label' => 'Requirements',
                'completed' => $requirementsPassed,
                'current' => $currentStep === 'requirements',
            ],
            [
                'key' => 'dependencies',
                'label' => 'Dependencies',
                'completed' => $dependenciesPassed,
                'current' => $currentStep === 'dependencies',
            ],
            [
                'key' => 'database',
                'label' => 'Database',
                'completed' => $databaseVerified,
                'current' => $currentStep === 'database',
            ],
            [
                'key' => 'application',
                'label' => 'Settings',
                'completed' => $applicationConfigured,
                'current' => $currentStep === 'application',
            ],
            [
                'key' => 'administrator',
                'label' => 'Create your Account',
                'completed' => $administratorCreated,
                'current' => $currentStep === 'administrator',
            ],
        ];
    }
}