<?php

namespace App\Livewire\Extensions;

use App\Extensions\ExtensionManager;
use App\Models\Extension;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Components\KeyValue;
use Filament\Notifications\Notification;
use Filament\Actions\{Action, BulkAction};
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Tables\Columns\{TextColumn, IconColumn};
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\HtmlString;
use Livewire\Component;

class Manage extends Component implements HasTable, HasForms, HasActions
{
    use InteractsWithTable;
    use InteractsWithForms;
    use InteractsWithActions;

    protected ExtensionManager $extensionManager;

    // Store extension ID for retry installation
    public ?string $pendingInstallExtensionId = null;

    public function boot(ExtensionManager $extensionManager): void
    {
        $this->extensionManager = $extensionManager;
    }

    public function table(Table $table): Table
    {
        // Discover extensions first
        $this->extensionManager->discover();

        // Get all discovered extensions
        $discoveredExtensions = $this->extensionManager->all();

        // Sync extension data to database for table display
        foreach ($discoveredExtensions as $extension) {
            $existing = Extension::where('extension_id', $extension->getId())->first();

            Extension::updateOrCreate(
                ['extension_id' => $extension->getId()],
                [
                    'name' => $extension->getName(),
                    'version' => $extension->getVersion(),
                    'installed' => $this->extensionManager->isInstalled($extension->getId()),
                    'enabled' => $this->extensionManager->isEnabled($extension->getId()),
                    // Only set settings if not already set (preserve user settings)
                    'settings' => $existing?->settings ?? [],
                ]
            );
        }

        return $table
            ->query(Extension::query())
            ->columns([
                TextColumn::make('name')
                    ->label('Extension')
                    ->description(function ($record) {
                        $description = $this->getExtensionDescription($record);
                        $homepage = $this->getExtensionHomepage($record);

                        if (!empty($homepage)) {
                            $description .= ' <a href="' . e($homepage) . '" target="_blank" rel="noopener noreferrer" class="text-primary-500 hover:text-primary-600 dark:text-primary-400" title="View homepage">🔗</a>';
                        }

                        // Wrap in a div with custom class for easier mobile hiding
                        return new HtmlString('<div class="extension-description-wrapper">' . $description . '</div>');
                    })
                    ->searchable()
                    ->sortable()
                    ->icon(fn ($record) => $this->getExtensionIcon($record))
                    ->iconColor('primary')
                    ->size('lg')
                    ->weight('bold')
                    ->toggleable(false),

                TextColumn::make('version')
                    ->label('Version')
                    ->badge()
                    ->sortable(),

                TextColumn::make('extension_id')
                    ->label('Author')
                    ->formatStateUsing(fn ($record) => $this->getExtensionAuthor($record)),

                IconColumn::make('extension_id')
                    ->label('Compatible')
                    ->state(fn ($record) => $this->getExtensionCompatible($record))
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->tooltip(fn ($record) => $this->getCompatibilityTooltip($record)),

                IconColumn::make('extension_id')
                    ->label('Installed')
                    ->state(fn ($record) => $this->getExtensionInstalled($record))
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('gray')
                    ->toggleable(false),

                IconColumn::make('enabled')
                    ->label('Enabled')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('gray')
                    ->toggleable(false),
            ])
            ->actions([
                Action::make('install')
                    ->label('Install')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->color('primary')
                    ->button()
                    ->visible(fn ($record) => !$this->getExtensionInstalled($record) && $this->getExtensionCompatible($record))
                    ->requiresConfirmation()
                    ->action(function ($record, $action) {
                        // First check permissions for LiveChat extension
                        if ($record->extension_id === 'live-chat') {
                            $publicPath = public_path('extension-assets/live-chat');

                            // Check if directory exists and is writable
                            if (!\Illuminate\Support\Facades\File::isDirectory($publicPath)) {
                                // Try to create directory
                                try {
                                    \Illuminate\Support\Facades\File::makeDirectory($publicPath, 0755, true);
                                } catch (\Exception $e) {
                                    // Store extension ID for retry
                                    $this->pendingInstallExtensionId = $record->extension_id;
                                    // Show permissions error notification with continue button
                                    $this->showPermissionsErrorModal($publicPath);
                                    return;
                                }
                            }
                        }

                        // Proceed with installation
                        $this->installExtension($record);
                    })
                    ->modalHeading(fn ($record) => 'Install ' . $this->getExtensionName($record))
                    ->modalDescription(fn ($record) => 'This will install the extension and run any necessary migrations.'),

                Action::make('enable')
                    ->label('Enable')
                    ->icon('heroicon-o-check-circle')
                    ->color('success')
                    ->button()
                    ->visible(fn ($record) => $this->getExtensionInstalled($record) && !$this->getExtensionEnabled($record))
                    ->action(fn ($record) => $this->enableExtension($record)),

                Action::make('disable')
                    ->label('Disable')
                    ->icon('heroicon-o-x-circle')
                    ->color('danger')
                    ->button()
                    ->visible(fn ($record) => $this->getExtensionEnabled($record))
                    ->requiresConfirmation()
                    ->action(fn ($record) => $this->disableExtension($record)),

                Action::make('uninstall')
                    ->label('Uninstall')
                    ->icon('heroicon-o-trash')
                    ->color('danger')
                    ->button()
                    ->visible(fn ($record) => $this->getExtensionInstalled($record))
                    ->requiresConfirmation()
                    ->action(fn ($record) => $this->uninstallExtension($record))
                    ->modalHeading(fn ($record) => 'Uninstall ' . $this->getExtensionName($record))
                    ->modalDescription('This will remove the extension. Database tables will not be automatically dropped.'),

                Action::make('configure')
                    ->label('Configure')
                    ->icon('heroicon-o-cog-6-tooth')
                    ->color('info')
                    ->button()
                    ->visible(fn ($record) => $this->getExtensionInstalled($record) && $this->extensionHasConfig($record))
                    ->modalHeading(fn ($record) => 'Configure ' . $this->getExtensionName($record))
                    ->modalDescription('Update extension settings')
                    ->modalWidth('xl')
                    ->form(function ($record) {
                        // Get extension object to access manifest defaults
                        $extension = $this->extensionManager->get($record->extension_id);
                        $defaults = [];

                        // Load default config from extension.json
                        if ($extension) {
                            $manifestPath = $extension->getPath() . '/extension.json';
                            if (file_exists($manifestPath)) {
                                $manifest = json_decode(file_get_contents($manifestPath), true);
                                $defaults = $manifest['config'] ?? [];
                            }
                        }

                        // Get saved settings from database
                        $savedSettings = $record->settings ?? [];

                        // Ensure settings is an array
                        if (!is_array($savedSettings)) {
                            $savedSettings = [];
                        }

                        // Merge defaults with saved settings (saved settings override defaults)
                        $settings = array_merge($defaults, $savedSettings);

                        // Filter out any entries with empty keys (KeyValue component can't handle them)
                        $settings = array_filter($settings, fn($key) => $key !== '', ARRAY_FILTER_USE_KEY);

                        return [
                            KeyValue::make('settings')
                                ->label('Extension Settings')
                                ->keyLabel('Setting Name')
                                ->valueLabel('Value')
                                ->default($settings)
                                ->addActionLabel('Add Setting')
                                ->reorderable()
                                ->deletable()
                                ->deleteAction(
                                    fn ($action) => $action
                                        ->color('gray')
                                        ->icon('heroicon-m-trash')
                                        ->iconButton()
                                )
                                ->editableKeys()
                                ->helperText('Configure key-value pairs for your extension. Common examples: api_key, webhook_url, timeout, debug_mode, etc.')
                                ->columnSpanFull(),
                        ];
                    })
                    ->action(function ($record, array $data) {
                        try {
                            // Filter out any entries with empty keys before saving
                            $settings = $data['settings'] ?? [];

                            // Ensure settings is an array
                            if (!is_array($settings)) {
                                $settings = [];
                            }

                            $settings = array_filter($settings, fn($key) => $key !== '', ARRAY_FILTER_USE_KEY);

                            // No need to json_encode - model cast handles it automatically
                            $record->settings = $settings;
                            $record->save();

                            Notification::make()
                                ->title('Settings Updated')
                                ->success()
                                ->body("Settings for '{$this->getExtensionName($record)}' have been updated.")
                                ->send();
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title('Update Failed')
                                ->danger()
                                ->body($e->getMessage())
                                ->send();
                        }
                    }),
            ])
            ->bulkActions([
                BulkAction::make('install')
                    ->label('Install Selected')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->color('primary')
                    ->requiresConfirmation()
                    ->action(function (Collection $records) {
                        foreach ($records as $record) {
                            if (!$this->getExtensionInstalled($record) && $this->getExtensionCompatible($record)) {
                                $this->installExtension($record);
                            }
                        }
                    }),

                BulkAction::make('enable')
                    ->label('Enable Selected')
                    ->icon('heroicon-o-check-circle')
                    ->color('success')
                    ->action(function (Collection $records) {
                        foreach ($records as $record) {
                            if ($this->getExtensionInstalled($record) && !$this->getExtensionEnabled($record)) {
                                $this->enableExtension($record);
                            }
                        }
                    }),

                BulkAction::make('disable')
                    ->label('Disable Selected')
                    ->icon('heroicon-o-x-circle')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->action(function (Collection $records) {
                        foreach ($records as $record) {
                            if ($this->getExtensionEnabled($record)) {
                                $this->disableExtension($record);
                            }
                        }
                    }),
            ])
            ->emptyStateHeading('No Extensions Found')
            ->emptyStateDescription('Install extensions to add new features to Ticaga.')
            ->emptyStateIcon('heroicon-o-puzzle-piece');
    }

    // Helper methods to extract data from records
    protected function getExtensionId($record): string
    {
        return $record->extension_id;
    }

    protected function getExtensionName($record): string
    {
        return $record->name;
    }

    protected function getExtensionDescription($record): string
    {
        $extension = $this->extensionManager->get($record->extension_id);
        return $extension ? $extension->getDescription() : '';
    }

    protected function getExtensionIcon($record): ?string
    {
        $extension = $this->extensionManager->get($record->extension_id);
        return $extension ? $extension->getIcon() : 'heroicon-o-puzzle-piece';
    }

    protected function getExtensionAuthor($record): string
    {
        $extension = $this->extensionManager->get($record->extension_id);
        return $extension ? $extension->getAuthor() : '';
    }

    protected function getExtensionHomepage($record): ?string
    {
        return $this->extensionManager->get($record->extension_id)?->getHomepage();
    }

    protected function getExtensionInstalled($record): bool
    {
        return $this->extensionManager->isInstalled($record->extension_id);
    }

    protected function getExtensionEnabled($record): bool
    {
        return $record->enabled ?? false;
    }

    protected function extensionHasConfig($record): bool
    {
        // Check if extension has saved settings
        if (!empty($record->settings) && is_array($record->settings) && count($record->settings) > 0) {
            return true;
        }

        // Check if extension has config section in manifest (extension.json)
        $extension = $this->extensionManager->get($record->extension_id);
        if ($extension) {
            $manifestPath = $extension->getPath() . '/extension.json';
            if (file_exists($manifestPath)) {
                $manifest = json_decode(file_get_contents($manifestPath), true);
                return !empty($manifest['config']);
            }
        }

        return false;
    }

    protected function getExtensionCompatible($record): bool
    {
        $extension = $this->extensionManager->get($record->extension_id);
        return $extension ? $extension->isCompatible() : false;
    }

    protected function getCompatibilityTooltip($record): string
    {
        if ($this->getExtensionCompatible($record)) {
            return 'Compatible with this version of Ticaga';
        }
        return 'Not compatible with this version of Ticaga';
    }

    // Extension actions
    protected function showPermissionsErrorModal($publicPath): void
    {
        $extensionAssetsPath = public_path('extension-assets');
        $currentUser = get_current_user();

        $commands = "mkdir -p {$extensionAssetsPath}\n";
        $commands .= "chmod -R 775 {$extensionAssetsPath}\n";
        $commands .= "chown -R {$currentUser}:apache {$extensionAssetsPath}";

        // Escape for JavaScript
        $commandsEscaped = addslashes($commands);

        // Show Filament notification with detailed error and action buttons
        Notification::make('permissions-error')
            ->title('Permission Issue Detected')
            ->warning()
            ->body(new HtmlString("
                <style>
                    .fi-no-permissions-error { max-width: 650px !important; }
                </style>
                <div class='space-y-3 fi-no-permissions-error'>
                    <p class='text-sm font-medium'>Failed to create directory:</p>
                    <div class='bg-gray-100 dark:bg-gray-800 rounded px-3 py-2'>
                        <code class='text-xs font-mono break-all'>{$publicPath}</code>
                    </div>

                    <div>
                        <div class='flex items-center justify-between mb-2'>
                            <p class='text-sm font-semibold text-gray-700 dark:text-gray-300'>Please run the following commands:</p>
                            <button
                                onclick=\"navigator.clipboard.writeText('{$commandsEscaped}').then(() => {
                                    const btn = this;
                                    const originalText = btn.innerHTML;
                                    btn.innerHTML = '<svg class=\\\"w-4 h-4 inline mr-1\\\" fill=\\\"currentColor\\\" viewBox=\\\"0 0 20 20\\\"><path fill-rule=\\\"evenodd\\\" d=\\\"M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z\\\" clip-rule=\\\"evenodd\\\"></path></svg>Copied!';
                                    setTimeout(() => { btn.innerHTML = originalText; }, 2000);
                                })\"
                                class='inline-flex items-center px-2 py-1 text-xs font-medium text-indigo-700 dark:text-indigo-300 bg-indigo-100 dark:bg-indigo-900/30 rounded hover:bg-indigo-200 dark:hover:bg-indigo-900/50 focus:outline-none focus:ring-2 focus:ring-indigo-500 transition-colors'>
                                <svg class='w-4 h-4 mr-1' fill='none' stroke='currentColor' viewBox='0 0 24 24'>
                                    <path stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z'></path>
                                </svg>
                                Copy Commands
                            </button>
                        </div>
                        <div class='bg-gray-900 dark:bg-gray-950 rounded-lg p-3 overflow-x-auto border border-gray-700'>
                            <pre class='text-xs text-green-400 font-mono leading-relaxed'>{$commands}</pre>
                        </div>
                    </div>

                    <div class='bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-2.5'>
                        <p class='text-xs text-blue-800 dark:text-blue-200'>
                            <strong>Options:</strong> Fix permissions and retry, or continue anyway (extension may not work correctly without proper permissions).
                        </p>
                    </div>

                    <div class='flex gap-2 mt-4'>
                        <button
                            wire:click=\"continueInstallation\"
                            class='inline-flex items-center px-3 py-2 text-sm font-semibold text-white bg-indigo-600 rounded-lg hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition-colors'>
                            Continue Anyway
                        </button>
                        <button
                            wire:click=\"dismissPermissionsError\"
                            class='inline-flex items-center px-3 py-2 text-sm font-semibold text-gray-700 dark:text-gray-200 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2 transition-colors'>
                            Cancel
                        </button>
                    </div>
                </div>
            "))
            ->persistent()
            ->duration(null)
            ->send();
    }

    protected function installExtension($record): void
    {
        try {
            $extensionId = $this->getExtensionId($record);
            $this->extensionManager->install($extensionId);

            Notification::make()
                ->title('Extension Installed')
                ->success()
                ->body("The {$this->getExtensionName($record)} extension has been installed successfully.")
                ->send();
        } catch (\Exception $e) {
            Notification::make()
                ->title('Installation Failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    protected function enableExtension($record): void
    {
        try {
            $extensionId = $this->getExtensionId($record);
            $this->extensionManager->enable($extensionId);

            Notification::make()
                ->title('Extension Enabled')
                ->success()
                ->body("The {$this->getExtensionName($record)} extension has been enabled.")
                ->send();
        } catch (\Exception $e) {
            Notification::make()
                ->title('Enable Failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    protected function disableExtension($record): void
    {
        try {
            $extensionId = $this->getExtensionId($record);
            $this->extensionManager->disable($extensionId);

            Notification::make()
                ->title('Extension Disabled')
                ->success()
                ->body("The {$this->getExtensionName($record)} extension has been disabled.")
                ->send();
        } catch (\Exception $e) {
            Notification::make()
                ->title('Disable Failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    protected function uninstallExtension($record): void
    {
        try {
            $extensionId = $this->getExtensionId($record);
            $this->extensionManager->uninstall($extensionId);

            Notification::make()
                ->title('Extension Uninstalled')
                ->success()
                ->body("The {$this->getExtensionName($record)} extension has been uninstalled.")
                ->send();
        } catch (\Exception $e) {
            Notification::make()
                ->title('Uninstall Failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    // Public Livewire methods for notification buttons
    public function continueInstallation(): void
    {
        if (!$this->pendingInstallExtensionId) {
            return;
        }

        // Find the extension record
        $record = Extension::where('extension_id', $this->pendingInstallExtensionId)->first();

        if (!$record) {
            Notification::make()
                ->title('Error')
                ->danger()
                ->body('Extension not found.')
                ->send();
            return;
        }

        // Proceed with installation (success notification will replace the warning)
        $this->installExtension($record);

        // Clear pending installation
        $this->pendingInstallExtensionId = null;
    }

    public function dismissPermissionsError(): void
    {
        // Clear pending installation and show dismissal message
        $this->pendingInstallExtensionId = null;

        Notification::make()
            ->title('Installation Cancelled')
            ->body('You can install the extension later after fixing permissions.')
            ->send();
    }

    public function render(): View
    {
        return view('livewire.extensions.manage');
    }
}
