<?php

namespace App\Livewire\Settings\Categories;

use App\Models\Categories;
use App\Enums\Type;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;

#[Layout("components.layouts.app", ["title" => "Create Category"])]
class Create extends Component
{
    public $category_name = "";
    public $uri = "";
    public $type = "knowledgebase"; // Default type
    public $display = 1;
    public $auto_generate_uri = true;
    public $parent_id = null;
    public $sort_order = 0;

    protected $rules = [
        "category_name" => "required|string|max:255",
        "uri" => "required|string|max:255|unique:categories,uri",
        "type" => "required|string",
        "display" => "required|boolean",
        "parent_id" => "nullable|exists:categories,id",
        "sort_order" => "nullable|integer|min:0",
    ];

    protected $messages = [
        "category_name.required" => "Category name is required.",
        "uri.required" => "URL slug is required.",
        "uri.unique" => "This URL slug is already in use.",
        "type.required" => "Category type is required.",
        "parent_id.exists" => "Selected parent category does not exist.",
    ];

    public function mount()
    {
        // Check if parent parameter was passed (for "Add Sub-category" action)
        $this->parent_id = request("parent");

        // Check if type parameter was passed
        if (request("type")) {
            $this->type = request("type");
        }

        // If parent is specified, inherit its type
        if ($this->parent_id) {
            $parent = Categories::find($this->parent_id);
            if ($parent) {
                // Convert enum to string value
                $this->type = $parent->type instanceof \BackedEnum ? $parent->type->value : $parent->type;
            }
        }

        // Validate that knowledgebase type is only allowed if extension is installed
        if ($this->type === 'knowledgebase') {
            $knowledgebaseExtension = \App\Models\Extension::where('extension_id', 'knowledgebase')
                ->where('enabled', true)
                ->first();

            if (!$knowledgebaseExtension) {
                // Extension not installed, default to announcements
                $this->type = 'announcements';
                session()->flash('warning', 'Knowledge Base extension is not installed. Category type changed to Announcements.');
            }
        }
    }

    public function updatedCategoryName()
    {
        if ($this->auto_generate_uri) {
            $this->uri = Str::slug($this->category_name);
        }
    }

    public function updatedUri()
    {
        $this->uri = Str::slug($this->uri);
    }

    public function toggleAutoGenerateUri()
    {
        $this->auto_generate_uri = !$this->auto_generate_uri;

        if ($this->auto_generate_uri && $this->category_name) {
            $this->uri = Str::slug($this->category_name);
        }
    }

    public function save()
    {
        $this->validate();

        // Validate parent relationship if parent is selected
        if ($this->parent_id) {
            $tempCategory = new Categories([
                "type" => $this->type,
            ]);

            $validation = $tempCategory->canHaveParent($this->parent_id, 3);

            if (!$validation["valid"]) {
                session()->flash("error", $validation["error"]);
                return;
            }
        }

        Categories::create([
            "category_name" => $this->category_name,
            "uri" => $this->uri,
            "type" => $this->type,
            "display" => $this->display,
            "parent_id" => $this->parent_id ?: null,
            "sort_order" => $this->sort_order,
        ]);

        $message = $this->parent_id
            ? "Sub-category created successfully!"
            : "Category created successfully!";

        session()->flash("success", $message);

        return redirect()->route("settings.categories");
    }

    public function render()
    {
        // Get all categories of the same type for parent dropdown (exclude deep nested ones - max 2 levels)
        $availableParents = Categories::where("type", $this->type)
            ->with("parent")
            ->get()
            ->filter(function ($category) {
                return $category->getDepth() < 2; // Only allow up to 2 levels deep
            })
            ->sortBy("category_name");

        // Get available types
        $availableTypes = [
            "announcements" => "Announcements",
        ];

        // Only show Knowledge Base if extension is installed and enabled
        $knowledgebaseExtension = \App\Models\Extension::where('extension_id', 'knowledgebase')
            ->where('enabled', true)
            ->first();

        if ($knowledgebaseExtension) {
            $availableTypes = [
                "knowledgebase" => "Knowledge Base",
            ] + $availableTypes;
        }

        return view("livewire.settings.categories.create", [
            "availableParents" => $availableParents,
            "availableTypes" => $availableTypes,
        ]);
    }
}
