<?php

    namespace App\Livewire\Tickets;

    use App\Http\Controllers\ResponsesController;
    use App\Models\{Tickets, Responses, Attachment};
    use App\Services\AttachmentService;

    use Illuminate\Support\Carbon;
    use Illuminate\Support\Facades\Auth;

    use Livewire\Attributes\Rule;
    use Livewire\Component;
    use Livewire\WithFileUploads;

    use Filament\Notifications\Notification;

    class Comments extends Component
    {
        use WithFileUploads;

        #[Rule('required', message: "We can't help you if you don't tell us what's wrong :(.")]
        public $content = '';
        public $response_type = [];
        public $files = [];
        public $settings = [];

        public $ticket_id;

        public function mount($ticket_id)
        {
            $this->ticket_id = $ticket_id;

            // Load settings for this ticket's department
            $ticket = Tickets::find($ticket_id);
            if ($ticket) {
                $this->settings = AttachmentService::getSettingsForDepartment($ticket->department_id);
                // Get count of existing attachments on this ticket
                $this->settings['current_file_count'] = Attachment::where('ticket_id', $ticket_id)->count();
            }
        }

        public function updatedFiles()
        {
            // Validate files on upload
            $this->validate([
                'files.*' => [
                    'file',
                    'max:' . ($this->settings['max_file_size_mb'] * 1024), // Convert MB to KB for validation
                ],
            ], [
                'files.*.max' => 'File size must not exceed ' . $this->settings['max_file_size_mb'] . 'MB.',
            ]);
        }

        public function send()
        {
            if ($this->response_type == '1') {
                $is_note = '1';
            } else {
                $is_note = '0';
                $check_priority = Tickets::where('id', '=', $this->ticket_id)->first();
                if ($check_priority->status != 'in progress') {
                    Tickets::where('id', '=', $this->ticket_id)->update(['status' => 'awaiting reply']);
                }
            }

            if (Auth::check() && Auth::user()->hasAnyRole('superadmin', 'admin', 'employee')) {
                $employee_response = '1';
            } else {
                $employee_response = '0';
                $check_priority = Tickets::where('id', '=', $this->ticket_id)->first();

                if ($check_priority->status != 'in progress') {
                    Tickets::where('id', '=', $this->ticket_id)->update(['status' => 'open']);
                }
            }

            $this->validate();

            $responseId = ResponsesController::store([
                'content' => $this->content,
                'ticket_number' => $this->ticket_id,
                'is_note' => $is_note,
                'employee_response' => $employee_response,
            ]);

            // Upload files if any were selected
            if (!empty($this->files) && $this->settings['enabled']) {
                $uploadedCount = 0;
                $errors = [];

                foreach ($this->files as $file) {
                    try {
                        AttachmentService::upload(
                            $file,
                            $this->ticket_id,
                            Auth::id() ?? $check_priority->user_id,
                            $responseId,
                            false
                        );
                        $uploadedCount++;
                    } catch (\Exception $e) {
                        $errors[] = $file->getClientOriginalName() . ': ' . $e->getMessage();
                    }
                }

                if (!empty($errors)) {
                    Notification::make()
                        ->title('Some files failed to upload')
                        ->warning()
                        ->body(implode("\n", $errors))
                        ->send();
                }
            }

            // Dispatch event to update sidebar counts
            $this->dispatch('ticket-updated')->to('sidebar-navigation');
            $this->dispatch('ticket-updated')->to('sidebar-assigned-tickets');

            if($is_note == '1')
            {
                Notification::make()
                    ->title('Note submitted successfully')
                    ->success()
                    ->body('Your private note has been added.')
                    ->color('success')
                    ->send();
            } else {
                Notification::make()
                    ->title('Response submitted successfully')
                    ->success()
                    ->body('Your response has been sent.')
                    ->color('success')
                    ->send();
            }

            if (Auth::check()) {
                $this->redirect('/tickets/show/'.$this->ticket_id);
            } else {
                $this->redirect('/tickets/public/'.$this->ticket_id.'/'.$check_priority->public_hash);
            }


            $this->reset();
        }

        public function getAcceptedFileTypes()
        {
            $allowed = $this->settings['allowed_file_types'] ?? [];

            if (empty($allowed)) {
                return '*';
            }

            // Convert extensions to accept attribute format
            $extensions = array_filter($allowed, fn($type) => !str_contains($type, '/'));

            return !empty($extensions) ? '.' . implode(',.', $extensions) : '*';
        }

        public function render()
        {
            return view('livewire.tickets.comments');
        }

    }
