<?php

use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Volt\Component;

new #[Layout('components.layouts.auth'), Title('Register')] class extends Component {
    public string $name = '';
    public string $email = '';
    public string $company = '';
    public string $phone_number = '';
    public string $password = '';
    public string $password_confirmation = '';

    /**
     * Handle an incoming registration request.
     */
    public function register(): void
    {
        $validated = $this->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'password' => ['required', 'string', 'confirmed', Password::min(15)->uncompromised()->letters()->numbers()->symbols()],
        ]);

        $validated['password'] = Hash::make($validated['password']);
        $validated['account_manager'] = '0';
        $validated['company'] = '';
        $validated['phone_number'] = '';
        $validated['biography'] = '';

        event(new Registered(($user = User::create($validated)->assignRole('customer'))));
        $api_token = $user->createToken('auth_token')->plainTextToken;
        $user->update(['api_token' => $api_token]);
        Auth::login($user);
        if(config('ticaga.registraton_logs') === true){
            Logs::create([
                'log_type' => 'notice',
                'message' => 'User ' . Auth::user()->name . ' has successfully registered.',
                'user_id' => Auth::id(),
                'customer_id' => Auth::id(),
                'visible' => '1',
                'ip_address' => request()->ip(),
            ]);
        }
        Logs::create([
            'log_type' => 'user',
            'message' => 'User authenticated successfully.',
            'user_id' => Auth::id(),
            'customer_id' => Auth::id(),
            'visible' => '1',
            'ip_address' => request()->ip(),
        ]);
        $this->redirectIntended(route('dashboard', absolute: false), navigate: true);
    }
}; ?>

<div class="flex flex-col gap-6">
    <x-auth-header :title="__('Create an account')" :description="__('Enter your details below to create your account')" />

    <!-- Session Status -->
    <x-auth-session-status class="text-center" :status="session('status')" />

    <form wire:submit="register" class="flex flex-col gap-6">
        <!-- Name -->
        <flux:input
            wire:model="name"
            :label="__('First and Last Name')"
            type="text"
            required
            autofocus
            autocomplete="name"
            :placeholder="__('Michael Taylor')"
        />

        <!-- Email Address -->
        <flux:input
            wire:model="email"
            :label="__('Email address')"
            type="email"
            required
            autocomplete="email"
            placeholder="mtaylor@example.com"
        />

        <!-- Password -->
        <flux:input
            wire:model="password"
            :label="__('Password')"
            type="password"
            required
            autocomplete="new-password"
            :placeholder="__('Password')"
        />

        <!-- Confirm Password -->
        <flux:input
            wire:model="password_confirmation"
            :label="__('Confirm password')"
            type="password"
            required
            autocomplete="new-password"
            :placeholder="__('Confirm password')"
        />

        <div class="flex items-center justify-end">
            <flux:button class="!bg-indigo-700 hover:!bg-indigo-500 !text-white w-full" type="submit">
                {{ __('Register') }}
            </flux:button>
        </div>
    </form>

    <div class="space-x-1 rtl:space-x-reverse text-center text-sm text-zinc-600 dark:text-zinc-400">
        {{ __('Already have an account?') }}
        <flux:link class="!text-indigo-700 hover:!text-indigo-500" :href="route('login')" wire:navigate>{{ __('Log in') }}</flux:link>
    </div>
</div>
