<div>
    @push('scripts')
        <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
        <script>
            (function() {
                const chartInstances = {};

                function parsePayload(value, fallback = null) {
                    if (!value) {
                        return fallback;
                    }

                    try {
                        return JSON.parse(value);
                    } catch (error) {
                        console.warn('SLA Reports: failed to parse chart payload', error);
                        return fallback;
                    }
                }

                function getChartPayload() {
                    const payloadEl = document.getElementById('sla-report-chart-payload');
                    if (!payloadEl) {
                        return null;
                    }

                    return {
                        department: parsePayload(payloadEl.dataset.department, { labels: [], compliance: [] }),
                        priority: parsePayload(payloadEl.dataset.priority, { labels: [], met: [], breached: [] }),
                        status: parsePayload(payloadEl.dataset.status, { labels: [], counts: [] })
                    };
                }

                function destroyChart(key) {
                    if (chartInstances[key]) {
                        chartInstances[key].destroy();
                        delete chartInstances[key];
                    }
                }

                function renderDepartmentChart(data) {
                    if (!data || !Array.isArray(data.labels)) {
                        destroyChart('department');
                        return;
                    }

                    const ctx = document.getElementById('departmentComplianceChart');
                    if (!ctx) {
                        return;
                    }

                    destroyChart('department');
                    chartInstances.department = new Chart(ctx, {
                        type: 'bar',
                        data: {
                            labels: data.labels,
                            datasets: [{
                                label: 'Compliance Rate (%)',
                                data: data.compliance || [],
                                backgroundColor: 'rgba(99, 102, 241, 0.5)',
                                borderColor: 'rgba(99, 102, 241, 1)',
                                borderWidth: 1,
                                borderRadius: 4
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    max: 100,
                                    ticks: {
                                        callback: value => value + '%'
                                    }
                                }
                            }
                        }
                    });
                }

                function renderPriorityChart(data) {
                    if (!data || !Array.isArray(data.labels)) {
                        destroyChart('priority');
                        return;
                    }

                    const ctx = document.getElementById('priorityOutcomeChart');
                    if (!ctx) {
                        return;
                    }

                    destroyChart('priority');
                    chartInstances.priority = new Chart(ctx, {
                        type: 'bar',
                        data: {
                            labels: data.labels,
                            datasets: [
                                {
                                    label: 'Met',
                                    data: data.met || [],
                                    backgroundColor: 'rgba(34, 197, 94, 0.6)',
                                    borderColor: 'rgba(34, 197, 94, 1)',
                                    borderWidth: 1
                                },
                                {
                                    label: 'Breached',
                                    data: data.breached || [],
                                    backgroundColor: 'rgba(239, 68, 68, 0.6)',
                                    borderColor: 'rgba(239, 68, 68, 1)',
                                    borderWidth: 1
                                }
                            ]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            scales: {
                                x: { stacked: true },
                                y: {
                                    beginAtZero: true,
                                    stacked: true
                                }
                            }
                        }
                    });
                }

                function renderStatusChart(data) {
                    if (!data || !Array.isArray(data.labels)) {
                        destroyChart('status');
                        return;
                    }

                    const ctx = document.getElementById('slaStatusChart');
                    if (!ctx) {
                        return;
                    }

                    destroyChart('status');
                    chartInstances.status = new Chart(ctx, {
                        type: 'doughnut',
                        data: {
                            labels: data.labels,
                            datasets: [{
                                data: data.counts || [],
                                backgroundColor: [
                                    'rgba(34, 197, 94, 0.6)',
                                    'rgba(239, 68, 68, 0.6)',
                                    'rgba(251, 191, 36, 0.6)',
                                    'rgba(59, 130, 246, 0.6)',
                                    'rgba(148, 163, 184, 0.6)'
                                ],
                                borderColor: [
                                    'rgba(34, 197, 94, 1)',
                                    'rgba(239, 68, 68, 1)',
                                    'rgba(251, 191, 36, 1)',
                                    'rgba(59, 130, 246, 1)',
                                    'rgba(148, 163, 184, 1)'
                                ],
                                borderWidth: 1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'bottom'
                                }
                            }
                        }
                    });
                }

                function renderCharts() {
                    if (typeof Chart === 'undefined') {
                        return;
                    }

                    const payload = getChartPayload();
                    if (!payload) {
                        return;
                    }

                    renderDepartmentChart(payload.department);
                    renderPriorityChart(payload.priority);
                    renderStatusChart(payload.status);
                }

                document.addEventListener('livewire:initialized', () => {
                    renderCharts();

                    Livewire.hook('message.processed', () => {
                        renderCharts();
                    });
                });
            })();
        </script>
    @endpush

    <div class="container mx-auto px-8 pb-8">
        <!-- Header -->
        <div class="flex items-center justify-between mb-6">
            <div>
                <h1 class="text-3xl font-bold text-gray-900 dark:text-white">SLA Reports & Analytics</h1>
                <p class="text-gray-600 dark:text-gray-400 mt-1">Comprehensive SLA performance metrics and trends</p>
            </div>
            <button wire:click="exportCsv" wire:loading.attr="disabled"
                class="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg transition disabled:opacity-50 flex items-center gap-2">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                </svg>
                <span wire:loading.remove>Export CSV</span>
                <span wire:loading>Exporting...</span>
            </button>
        </div>

        <!-- Filters -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6 mb-6">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Filters</h3>
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <!-- Time Period -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Time Period</label>
                    <select wire:model.live="period"
                        class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:ring-2 focus:ring-indigo-500">
                        <option value="last_7_days">Last 7 Days</option>
                        <option value="last_30_days">Last 30 Days</option>
                        <option value="last_3_months">Last 3 Months</option>
                        <option value="last_6_months">Last 6 Months</option>
                        <option value="this_year">This Year</option>
                    </select>
                </div>

                <!-- Department Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Department</label>
                    <select wire:model.live="departmentFilter"
                        class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:ring-2 focus:ring-indigo-500">
                        <option value="all">All Departments</option>
                        @foreach($departments as $dept)
                            <option value="{{ $dept->id }}">{{ $dept->department_name }}</option>
                        @endforeach
                    </select>
                </div>

                <!-- Priority Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Priority</label>
                    <select wire:model.live="priorityFilter"
                        class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:ring-2 focus:ring-indigo-500">
                        <option value="all">All Priorities</option>
                        <option value="low">Low</option>
                        <option value="medium">Medium</option>
                        <option value="high">High</option>
                        <option value="urgent">Urgent</option>
                    </select>
                </div>

                <!-- Employee Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Employee</label>
                    <select wire:model.live="employeeFilter"
                        class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:ring-2 focus:ring-indigo-500">
                        <option value="all">All Employees</option>
                        @foreach($employees as $emp)
                            <option value="{{ $emp->id }}">{{ $emp->name }}</option>
                        @endforeach
                    </select>
                </div>
            </div>
        </div>

        @php
            $departmentCollection = collect($reportData['by_department'] ?? []);
            $priorityCollection = collect($reportData['by_priority'] ?? []);
            $summaryCollection = $reportData['summary'] ?? [];

            $departmentChartData = [
                'labels' => $departmentCollection->pluck('department_name')->values(),
                'compliance' => $departmentCollection->pluck('compliance_rate')->map(fn($rate) => round((float) $rate, 2))->values(),
            ];

            $priorityChartData = [
                'labels' => $priorityCollection->pluck('priority')->map(fn($priority) => $priority ? ucfirst($priority) : 'Unspecified')->values(),
                'met' => $priorityCollection->pluck('met')->values(),
                'breached' => $priorityCollection->pluck('breached')->values(),
            ];

            $statusChartData = [
                'labels' => ['Met', 'Breached', 'At Risk', 'Paused', 'Exempt'],
                'counts' => [
                    $summaryCollection['met'] ?? 0,
                    $summaryCollection['breached'] ?? 0,
                    $summaryCollection['at_risk'] ?? 0,
                    $summaryCollection['paused'] ?? 0,
                    $summaryCollection['exempt'] ?? 0,
                ],
            ];
        @endphp

        <div id="sla-report-chart-payload"
            data-department='@json($departmentChartData)'
            data-priority='@json($priorityChartData)'
            data-status='@json($statusChartData)'
            class="hidden"
        ></div>

        <!-- Summary Cards -->
        <div class="grid grid-cols-2 md:grid-cols-6 gap-4 mb-6">
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">Total</p>
                <p class="text-2xl font-bold text-gray-900 dark:text-white">{{ $reportData['summary']['total'] ?? 0 }}</p>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">Met</p>
                <p class="text-2xl font-bold text-green-600 dark:text-green-400">{{ $reportData['summary']['met'] ?? 0 }}</p>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">Breached</p>
                <p class="text-2xl font-bold text-red-600 dark:text-red-400">{{ $reportData['summary']['breached'] ?? 0 }}</p>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">At Risk</p>
                <p class="text-2xl font-bold text-yellow-600 dark:text-yellow-400">{{ $reportData['summary']['at_risk'] ?? 0 }}</p>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">Paused</p>
                <p class="text-2xl font-bold text-blue-600 dark:text-blue-400">{{ $reportData['summary']['paused'] ?? 0 }}</p>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4">
                <p class="text-xs text-gray-600 dark:text-gray-400 mb-1">Exempt</p>
                <p class="text-2xl font-bold text-gray-600 dark:text-gray-400">{{ $reportData['summary']['exempt'] ?? 0 }}</p>
            </div>
        </div>

        <!-- Charts -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Compliance by Department</h3>
                <div class="h-64">
                    <canvas id="departmentComplianceChart"></canvas>
                </div>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Outcomes by Priority</h3>
                <div class="h-64">
                    <canvas id="priorityOutcomeChart"></canvas>
                </div>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">SLA Status Distribution</h3>
                <div class="h-64">
                    <canvas id="slaStatusChart"></canvas>
                </div>
            </div>
        </div>

        <!-- Compliance Rate -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6 mb-6">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Compliance Rate</h3>
            <div class="flex items-center">
                <div class="flex-1">
                    <div class="relative pt-1">
                        <div class="overflow-hidden h-6 text-xs flex rounded-full bg-gray-200 dark:bg-gray-700">
                            <div style="width:{{ $reportData['summary']['compliance_rate'] ?? 0 }}%"
                                 class="shadow-none flex flex-col text-center whitespace-nowrap text-white justify-center transition-all duration-500
                                 {{ ($reportData['summary']['compliance_rate'] ?? 0) >= 90 ? 'bg-green-500' : (($reportData['summary']['compliance_rate'] ?? 0) >= 70 ? 'bg-yellow-500' : 'bg-red-500') }}">
                            </div>
                        </div>
                    </div>
                </div>
                <span class="ml-4 text-3xl font-bold
                    {{ ($reportData['summary']['compliance_rate'] ?? 0) >= 90 ? 'text-green-600 dark:text-green-400' : (($reportData['summary']['compliance_rate'] ?? 0) >= 70 ? 'text-yellow-600 dark:text-yellow-400' : 'text-red-600 dark:text-red-400') }}">
                    {{ number_format($reportData['summary']['compliance_rate'] ?? 0, 1) }}%
                </span>
            </div>
        </div>

        <!-- Department Performance Table -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6 mb-6">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Performance by Department</h3>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-900">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Department</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Total</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Met</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Breached</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Compliance</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        @forelse($reportData['by_department'] ?? [] as $dept)
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                {{ $dept->department_name }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                {{ $dept->total }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400">
                                {{ $dept->met }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-red-600 dark:text-red-400">
                                {{ $dept->breached }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm">
                                <span class="px-2 py-1 rounded-full text-xs font-medium
                                    {{ $dept->compliance_rate >= 90 ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' :
                                       ($dept->compliance_rate >= 70 ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200' :
                                       'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200') }}">
                                    {{ number_format($dept->compliance_rate, 1) }}%
                                </span>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="5" class="px-6 py-4 text-center text-gray-500 dark:text-gray-400">No data available</td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Employee Performance Table -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Top 10 Employee Performance</h3>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-900">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Rank</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Employee</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Total</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Met</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Breached</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Compliance</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        @forelse($reportData['by_employee'] ?? [] as $index => $emp)
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                {{ $index + 1 }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                {{ $emp->employee_name }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                {{ $emp->total }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400">
                                {{ $emp->met }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-red-600 dark:text-red-400">
                                {{ $emp->breached }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm">
                                <span class="px-2 py-1 rounded-full text-xs font-medium
                                    {{ $emp->compliance_rate >= 90 ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' :
                                       ($emp->compliance_rate >= 70 ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200' :
                                       'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200') }}">
                                    {{ number_format($emp->compliance_rate, 1) }}%
                                </span>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="6" class="px-6 py-4 text-center text-gray-500 dark:text-gray-400">No data available</td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
