<?php

use App\Models\Categories;
use App\Models\Extension;
use App\Enums\Type;
use Illuminate\Support\Str;
use Livewire\Volt\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;

new #[Title('Categories')] class extends Component {
    public $categories = [];
    public $showCreateModal = false;
    public $showEditModal = false;
    public $showDeleteModal = false;

    // Form properties
    public $categoryId = null;
    public $category_name = '';
    public $uri = '';
    public $type = 'announcements';
    public $display = true;

    /**
     * Mount the component.
     */
    public function mount(): void
    {
        $this->loadCategories();
    }

    /**
     * Get available category types based on installed extensions
     */
    public function getCategoryTypes(): array
    {
        $types = [];

        // Announcements (always available - core feature)
        $types['announcements'] = [
            'value' => 'announcements',
            'label' => 'Announcements',
            'icon' => 'newspaper',
            'enabled' => true,
        ];

        // Knowledgebase (extension-based)
        if (Extension::where('extension_id', 'knowledgebase')
            ->where('installed', true)
            ->where('enabled', true)
            ->exists()) {
            $types['knowledgebase'] = [
                'value' => 'knowledgebase',
                'label' => 'Knowledgebase',
                'icon' => 'academic-cap',
                'enabled' => true,
            ];
        }

        return $types;
    }

    /**
     * Load all categories
     */
    public function loadCategories(): void
    {
        $this->categories = Categories::withCount(['announcements', 'knowledgebase'])
            ->orderBy('category_name')
            ->get();
    }

    /**
     * Open create modal
     */
    public function openCreateModal(): void
    {
        $this->reset(['categoryId', 'category_name', 'uri', 'type', 'display']);
        $this->type = 'announcements'; // Default to announcements
        $this->display = true;
        $this->showCreateModal = true;
    }

    /**
     * Open edit modal
     */
    public function openEditModal($categoryId): void
    {
        $category = Categories::findOrFail($categoryId);

        $this->categoryId = $category->id;
        $this->category_name = $category->category_name;
        $this->uri = $category->uri;
        $this->type = $category->type->value;
        $this->display = (bool) $category->display;

        $this->showEditModal = true;
    }

    /**
     * Create category
     */
    public function createCategory(): void
    {
        $this->validate([
            'category_name' => 'required|string|max:255',
            'uri' => 'nullable|string|max:255|unique:categories,uri',
            'type' => 'required|in:announcements,knowledgebase',
            'display' => 'boolean',
        ]);

        // Generate URI if not provided (exactly like Departments/Announcements)
        if ($this->uri == '') {
            $this->uri = Str::of($this->category_name)->slug('-');
        }
        $this->uri = Str::of($this->uri)->slug('-');

        Categories::create([
            'category_name' => $this->category_name,
            'uri' => $this->uri,
            'type' => Type::from($this->type),
            'display' => $this->display,
        ]);

        session()->flash('success', 'Category created successfully.');

        $this->showCreateModal = false;
        $this->reset(['categoryId', 'category_name', 'uri', 'type', 'display']);
        $this->loadCategories();
    }

    /**
     * Update category
     */
    public function updateCategory(): void
    {
        $this->validate([
            'category_name' => 'required|string|max:255',
            'uri' => 'nullable|string|max:255|unique:categories,uri,' . $this->categoryId,
            'type' => 'required|in:announcements,knowledgebase',
            'display' => 'boolean',
        ]);

        // Generate URI if not provided (exactly like Departments/Announcements)
        if ($this->uri == '') {
            $this->uri = Str::of($this->category_name)->slug('-');
        }
        $this->uri = Str::of($this->uri)->slug('-');

        $category = Categories::findOrFail($this->categoryId);

        $category->update([
            'category_name' => $this->category_name,
            'uri' => $this->uri,
            'type' => Type::from($this->type),
            'display' => $this->display,
        ]);

        session()->flash('success', 'Category updated successfully.');

        $this->showEditModal = false;
        $this->reset(['categoryId', 'category_name', 'uri', 'type', 'display']);
        $this->loadCategories();
    }

    /**
     * Confirm delete
     */
    public function confirmDelete($categoryId): void
    {
        $this->categoryId = $categoryId;
        $this->showDeleteModal = true;
    }

    /**
     * Delete category
     */
    public function deleteCategory(): void
    {
        $category = Categories::findOrFail($this->categoryId);

        // Check if category has any content
        $announcementCount = $category->announcements()->count();
        $knowledgebaseCount = $category->knowledgebase()->count();

        if ($announcementCount > 0 || $knowledgebaseCount > 0) {
            session()->flash('error', 'Cannot delete category with existing content. Please reassign or delete content first.');
            $this->showDeleteModal = false;
            return;
        }

        $category->delete();

        session()->flash('success', 'Category deleted successfully.');

        $this->showDeleteModal = false;
        $this->categoryId = null;
        $this->loadCategories();
    }

    /**
     * Cancel delete
     */
    public function cancelDelete(): void
    {
        $this->showDeleteModal = false;
        $this->categoryId = null;
    }

    /**
     * Get content count for a category
     */
    public function getContentCount($category): string
    {
        $counts = [];

        if ($category->announcements_count > 0) {
            $counts[] = $category->announcements_count . ' ' . Str::plural('announcement', $category->announcements_count);
        }

        if ($category->knowledgebase_count > 0) {
            $counts[] = $category->knowledgebase_count . ' ' . Str::plural('article', $category->knowledgebase_count);
        }

        return !empty($counts) ? implode(', ', $counts) : 'No content';
    }

    /**
     * Get type label
     */
    public function getTypeLabel($type): string
    {
        $types = $this->getCategoryTypes();
        return $types[$type]['label'] ?? ucfirst($type);
    }

    /**
     * Get type icon
     */
    public function getTypeIcon($type): string
    {
        $types = $this->getCategoryTypes();
        return $types[$type]['icon'] ?? 'folder';
    }
}; ?>

<section class="w-full">
    @include('partials.settings-heading')

    <x-settings.layout
        :heading="__('Categories')"
        :subheading="__('Manage categories for announcements, knowledgebase, and other content')"
        :fullWidth="true"
    >
        {{-- Flash Messages --}}
        @if (session()->has('success'))
            <div class="mb-4 bg-green-50 dark:bg-green-900 border-l-4 border-green-400 p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <svg class="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                        </svg>
                    </div>
                    <div class="ml-3">
                        <div class="text-sm text-green-700 dark:text-green-200">
                            {{ session('success') }}
                        </div>
                    </div>
                </div>
            </div>
        @endif

        @if (session()->has('error'))
            <div class="mb-4 bg-red-50 dark:bg-red-900 border-l-4 border-red-400 p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                        </svg>
                    </div>
                    <div class="ml-3">
                        <div class="text-sm text-red-700 dark:text-red-200">
                            {{ session('error') }}
                        </div>
                    </div>
                </div>
            </div>
        @endif

        {{-- Toolbar --}}
        <div class="flex justify-end mb-6">
            <flux:button
                wire:click="openCreateModal"
                icon="plus"
                class="!bg-indigo-700 hover:!bg-indigo-500 !text-white"
            >
                Create Category
            </flux:button>
        </div>

        {{-- Categories Table --}}
        @if(count($categories) > 0)
            <div class="overflow-x-auto bg-white dark:bg-gray-800 rounded-lg shadow">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-900">
                        <tr>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                Category Name
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                Type
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                URL Slug
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                Content
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                Status
                            </th>
                            <th scope="col" class="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        @foreach($categories as $category)
                            <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center gap-2">
                                        <x-ticaga-icon :name="$this->getTypeIcon($category->type->value)" class="w-5 h-5 text-gray-400" />
                                        <div class="text-sm font-medium text-gray-900 dark:text-gray-100">
                                            {{ $category->category_name }}
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <flux:badge color="zinc" size="sm">
                                        {{ $this->getTypeLabel($category->type->value) }}
                                    </flux:badge>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-500 dark:text-gray-400">
                                        {{ $category->uri }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900 dark:text-gray-100">
                                        {{ $this->getContentCount($category) }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    @if($category->display)
                                        <flux:badge color="green" size="sm">Visible</flux:badge>
                                    @else
                                        <flux:badge color="gray" size="sm">Hidden</flux:badge>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-2">
                                    <button
                                        wire:click="openEditModal({{ $category->id }})"
                                        class="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                    >
                                        Edit
                                    </button>
                                    <button
                                        wire:click="confirmDelete({{ $category->id }})"
                                        class="text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                    >
                                        Delete
                                    </button>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        @else
            {{-- Empty State --}}
            <div class="text-center py-12 bg-gray-50 dark:bg-gray-800 rounded-lg">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z" />
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">No categories found</h3>
                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">
                    Get started by creating your first category.
                </p>
            </div>
        @endif

        {{-- Create Modal --}}
        <flux:modal wire:model="showCreateModal" name="create-category-modal" variant="flyout">
            <div>
                <flux:heading size="lg">Create Category</flux:heading>
                <flux:subheading class="mt-2">
                    Add a new category for organizing your content.
                </flux:subheading>

                <form wire:submit="createCategory" class="mt-6 space-y-6">
                        <div>
                            <flux:input
                                wire:model="category_name"
                                label="Category Name"
                                placeholder="Enter category name"
                                required
                            />
                            @error('category_name')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <flux:input
                                wire:model="uri"
                                label="URL Slug"
                                badge="Optional"
                                placeholder="url-friendly-slug"
                            />
                            @error('uri')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <flux:select wire:model="type" label="Type" required>
                                @foreach($this->getCategoryTypes() as $typeOption)
                                    <option value="{{ $typeOption['value'] }}">{{ $typeOption['label'] }}</option>
                                @endforeach
                            </flux:select>
                            @error('type')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <label class="flex items-center gap-2 cursor-pointer">
                                <input
                                    type="checkbox"
                                    wire:model="display"
                                    class="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                                <span class="text-sm text-gray-700 dark:text-gray-300">Visible to public</span>
                            </label>
                        </div>

                        <div class="mt-6 flex justify-end space-x-3">
                            <flux:button wire:click="$set('showCreateModal', false)" variant="ghost">
                                Cancel
                            </flux:button>
                            <flux:button type="submit" class="!bg-indigo-700 hover:!bg-indigo-500 !text-white" wire:loading.attr="disabled" wire:loading.class="opacity-50">
                                <span wire:loading.remove wire:target="createCategory">Create Category</span>
                                <span wire:loading wire:target="createCategory">Creating...</span>
                            </flux:button>
                        </div>
                    </form>
                </div>
        </flux:modal>

        {{-- Edit Modal --}}
        <flux:modal wire:model="showEditModal" name="edit-category-modal" variant="flyout">
            <div>
                <flux:heading size="lg">Edit Category</flux:heading>
                <flux:subheading class="mt-2">
                    Update category information.
                </flux:subheading>

                <form wire:submit="updateCategory" class="mt-6 space-y-6">
                        <div>
                            <flux:input
                                wire:model="category_name"
                                label="Category Name"
                                placeholder="Enter category name"
                                required
                            />
                            @error('category_name')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <flux:input
                                wire:model="uri"
                                label="URL Slug (optional - auto-generated if empty)"
                                placeholder="url-friendly-slug"
                            />
                            @error('uri')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <flux:select wire:model="type" label="Type" required>
                                @foreach($this->getCategoryTypes() as $typeOption)
                                    <option value="{{ $typeOption['value'] }}">{{ $typeOption['label'] }}</option>
                                @endforeach
                            </flux:select>
                            @error('type')
                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                            @enderror
                        </div>

                        <div>
                            <label class="flex items-center gap-2 cursor-pointer">
                                <input
                                    type="checkbox"
                                    wire:model="display"
                                    class="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                                />
                                <span class="text-sm text-gray-700 dark:text-gray-300">Visible to public</span>
                            </label>
                        </div>

                        <div class="mt-6 flex justify-end space-x-3">
                            <flux:button wire:click="$set('showEditModal', false)" variant="ghost">
                                Cancel
                            </flux:button>
                            <flux:button type="submit" class="!bg-indigo-700 hover:!bg-indigo-500 !text-white" wire:loading.attr="disabled" wire:loading.class="opacity-50">
                                <span wire:loading.remove wire:target="updateCategory">Update Category</span>
                                <span wire:loading wire:target="updateCategory">Updating...</span>
                            </flux:button>
                        </div>
                    </form>
                </div>
        </flux:modal>

        {{-- Delete Confirmation Modal --}}
        <flux:modal wire:model="showDeleteModal" name="delete-category-modal" variant="flyout">
            <div>
                <flux:heading size="lg">Confirm Deletion</flux:heading>
                <flux:subheading class="mt-2">
                    Are you sure you want to delete this category? This action cannot be undone.
                </flux:subheading>

                    <div class="mt-6 flex justify-end space-x-3">
                        <flux:button wire:click="cancelDelete" variant="ghost">
                            Cancel
                        </flux:button>
                        <flux:button wire:click="deleteCategory" variant="danger">
                            Delete Category
                        </flux:button>
                    </div>
                </div>
        </flux:modal>
    </x-settings.layout>
</section>
